package okrrpclogic

import (
	"context"
	"encoding/json"
	"errors"
	"time"

	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type CreateKeyResultLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewCreateKeyResultLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreateKeyResultLogic {
	return &CreateKeyResultLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 创建KR
func (l *CreateKeyResultLogic) CreateKeyResult(in *oa.CreateKeyResultReq) (*oa.CreateKeyResultResp, error) {
	deadline, err := time.Parse(time.DateOnly, in.GetDeadline())
	if err != nil {
		return nil, status.Error(codes.InvalidArgument, "无效的deadline")
	}

	myOkr, err := l.svcCtx.OkrPeriodModel.FindOne(l.ctx, in.GetOkrId())
	if err != nil {
		if errors.Is(err, okr.ErrNotFound) {
			return nil, ErrOkrNotFound
		}
		logx.Errorw("failed to query okr_period on CreateKeyResult", logx.Field("error", err))
		return nil, ErrOkrPeriodQueryFailed
	}
	if !IsOkrEditable(myOkr, in.GetOwner()) {
		logx.Infow("This okr is not editable", logx.Field("period_id", myOkr.PeriodId))
		return nil, ErrPermissionDenied
	}

	objectiveList, err := l.svcCtx.OkrTaskModel.GetByOkrPeriodIdAndEntityType(myOkr.PeriodId, OkrEntityTypeObjective)
	if err != nil {
		logx.Errorw("failed to query objective on CreateKeyResult", logx.Field("error", err))
		return nil, ErrOkrTaskQueryFailed
	}
	myObjective := objectiveList.GetByEntityId(in.GetObjectiveId())
	if myObjective == nil {
		logx.Infow("this objective doesn't belongs to you", logx.Field("period_id", myOkr.PeriodId), logx.Field("objective_id", in.GetObjectiveId()))
		return nil, ErrPermissionDenied
	}

	// 查询现有的KR
	krList, err := l.svcCtx.OkrTaskModel.GetByOkrPeriodIdAndEntityType(myOkr.PeriodId, OkrEntityTypeKeyresult)
	if err != nil {
		logx.Errorw("failed to query keyresult on CreateKeyResult", logx.Field("error", err))
		return nil, ErrOkrTaskQueryFailed
	}
	alignedKrList, err := l.svcCtx.OkrAlignmentModel.FindAllByUserAndAlignWithEntitiyId(l.ctx, in.GetOwner(), in.GetObjectiveId())
	if err != nil {
		logx.Errorw("failed to query x_okr_alignment on CreateKeyResult", logx.Field("error", err))
		return nil, ErrOkrAlignmentQueryFailed
	}
	// todo: add weight validation

	isAssignment := false
	pos := int64(len(krList) + len(alignedKrList) + 1)
	kr := okr.XOKrTask{
		EntityType:     OkrEntityTypeKeyresult,
		ParentEntityId: in.GetObjectiveId(),
		PeriodId:       myOkr.PeriodId,
		Content:        in.GetContent(),
		Weight:         in.GetWeight(),
		Priority:       in.GetPriority(),
		Deadline:       deadline,
		TypeForKr:      int64(in.GetTypeForKr()),
		Owner:          in.GetOwner(),
		Position:       pos,
	}
	if in.GetUserEmail() != in.GetOwner() {
		isAssignment = true
		kr.IntendedOwner = in.GetUserEmail()
	}
	err = l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, s sqlx.Session) error {
		insertResult, err := okr.NewXOKrTaskModel(sqlx.NewSqlConnFromSession(s)).Insert(l.ctx, &kr)
		if err != nil {
			logx.Errorw("failed to create keyresult on CreateKeyResult", logx.Field("error", err))
			return err
		}
		kr.EntityId, err = insertResult.LastInsertId()
		if err != nil {
			logx.Errorw("failed to get keyresult create.LastInsertId on CreateKeyResult", logx.Field("error", err))
			return err
		}

		if isAssignment { // 是分配KR
			msgContent := okr.MsgContentAssign{
				EntityId:   kr.EntityId,
				ReqBy:      in.GetOwner(),
				EntityPath: []*okr.XOKrTask{myObjective, &kr},
			}
			msgContentBytes, _ := json.Marshal(&msgContent)
			msg := okr.XOkrMsg{
				FromUser:       in.GetOwner(),
				ToUser:         in.GetUserEmail(),
				Type:           OkrMessgeTypeAssignReq,
				Content:        string(msgContentBytes),
				ApprovalStatus: OkrMessageApprovalStatusUnhandled,
				StartDate:      myOkr.StartDate,
				EndDate:        myOkr.EndDate,
			}
			msgInsertResult, err := l.svcCtx.OkrMsgModel.InsertWithSession(s, &msg)
			if err != nil {
				logx.Errorw("failed to create okr_message on CreateKeyResult", logx.Field("error", err))
				return err
			}
			_, err = msgInsertResult.LastInsertId()
			if err != nil {
				logx.Errorw("failed to get okr_message.LastInsertId on CreateKeyResult", logx.Field("error", err))
				return err
			}
		}

		return nil
	})
	if err != nil {
		return nil, ErrOkrTaskAddFailed
	}

	return &oa.CreateKeyResultResp{KeyresultId: kr.EntityId}, nil
}
