package okrrpclogic

import (
	"context"
	"errors"

	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type DeleteKeyResultLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewDeleteKeyResultLogic(ctx context.Context, svcCtx *svc.ServiceContext) *DeleteKeyResultLogic {
	return &DeleteKeyResultLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 删除KR
func (l *DeleteKeyResultLogic) DeleteKeyResult(in *oa.DeleteKeyResultReq) (*oa.DeleteKeyResultResp, error) {
	myOkr, err := l.svcCtx.OkrPeriodModel.FindOne(l.ctx, in.GetOkrId())
	if err != nil {
		if errors.Is(err, okr.ErrNotFound) {
			return nil, ErrOkrNotFound
		}
		logx.Errorw("failed to query okr_period on DeleteKeyResult", logx.Field("error", err))
		return nil, ErrOkrPeriodQueryFailed
	}
	if !IsOkrEditable(myOkr, in.GetOwner()) {
		logx.Infow("This okr is not editable", logx.Field("period_id", myOkr.PeriodId))
		return nil, ErrPermissionDenied
	}

	objectiveList, err := l.svcCtx.OkrTaskModel.GetByOkrPeriodIdAndEntityType(myOkr.PeriodId, OkrEntityTypeObjective)
	if err != nil {
		logx.Errorw("failed to query objective on DeleteKeyResult", logx.Field("error", err))
		return nil, ErrOkrTaskQueryFailed
	}
	myObjective := objectiveList.GetByEntityId(in.GetObjectiveId())
	if myObjective == nil {
		logx.Infow("this objective doesn't belongs to you", logx.Field("period_id", myOkr.PeriodId), logx.Field("objective_id", in.GetObjectiveId()))
		return nil, ErrPermissionDenied
	}

	// 自建KR，删除KR，再删除被对齐关系；
	// 来自分配的KR，删除KR，再删除双向对齐关系；
	// 来自对齐的KR，删除对齐关系；
	var alignRecord *okr.XOkrAlignment
	krList, err := l.svcCtx.OkrTaskModel.GetByParentEntityId(in.GetObjectiveId())
	if err != nil {
		logx.Errorw("failed to query keyresult on DeleteKeyResult", logx.Field("error", err))
		return nil, ErrOkrTaskQueryFailed
	}
	kr := krList.GetByEntityId(in.GetKeyresultId())
	if kr == nil {
		// 可能来自对齐
		alignRecord, err = l.svcCtx.OkrAlignmentModel.GetByEntityId(l.ctx, in.GetKeyresultId())
		if err != nil {
			if errors.Is(err, okr.ErrNotFound) {
				logx.Infow("this keyresult doesn't belongs to you", logx.Field("period_id", in.GetOkrId()),
					logx.Field("objective_id", in.GetObjectiveId()), logx.Field("keyresult_id", in.GetKeyresultId()))
				return nil, ErrPermissionDenied
			}
			logx.Errorw("failed to query x_okr_alignment on DeleteKeyResult", logx.Field("error", err))
			return nil, ErrOkrAlignmentQueryFailed
		}
	}

	err = l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, s sqlx.Session) error {
		if alignRecord != nil {
			// 来自对齐，只需要删除对齐记录即可；
			if err := l.svcCtx.OkrAlignmentModel.TransDelete(s, []int64{alignRecord.Id}); err != nil {
				logx.Errorw("failed to delete x_okr_alignment on DeleteKeyresult", logx.Field("error", err))
				return err
			}
			return nil
		}

		// 自建KR，来自分配KR；需要删除KR本向，再加上对齐的双向关系。
		if err := l.svcCtx.OkrTaskModel.TransDeleteCtx(ctx, s, []int64{in.GetKeyresultId()}); err != nil {
			logx.Errorw("failed to delete keyresult on DeleteKeyResult", logx.Field("error", err))
			return err
		}

		// 删除主动对齐关系
		if err := l.svcCtx.OkrAlignmentModel.DeleteByEntityId(ctx, s, in.GetKeyresultId()); err != nil {
			logx.Errorw("failed to delete x_okr_alignment on DeleteKeyResult-删除主动对齐关系", logx.Field("error", err))
			return err
		}

		// 删除被动对齐关系
		if err := l.svcCtx.OkrAlignmentModel.DeleteByAlignWithEntityIdAndUserEmail(ctx, s, in.GetKeyresultId(), in.GetOwner()); err != nil {
			logx.Errorw("failed to delete x_okr_alignment on DeleteKeyResult-删除被动对齐关系", logx.Field("error", err))
			return err
		}

		return nil
	})
	if err != nil {
		return nil, ErrOkrTaskDeleteFailed
	}

	return &oa.DeleteKeyResultResp{}, nil
}
