package okrrpclogic

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"time"

	"github.com/zeromicro/go-zero/core/logx"
)

type MessageAlignHandleLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewMessageAlignHandleLogic(ctx context.Context, svcCtx *svc.ServiceContext) *MessageAlignHandleLogic {
	return &MessageAlignHandleLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *MessageAlignHandleLogic) MessageAlignHandle(in *oa.MessageAlignHandleReq) (*oa.EmptyResponse, error) {
	var rtnErr error

	txErr := l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, session sqlx.Session) error {
		msg, err := okr.NewXOkrMsgModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, in.MessageId)
		if err != nil {
			return fmt.Errorf("NewXOkrMsgModel.FindOne: %w", err)
		}
		if msg.Type != 1 {
			return errors.New("这不是一个对齐请求消息")
		}
		if msg.ToUser != in.UserEmail {
			return errors.New("你没有处理权限")
		}
		if in.Status == 3 { //拒绝
			//do nothing
		} else if in.Status == 6 { //接受
			//do nothing
		} else {
			return errors.New("unknown status")
		}
		if msg.ApprovalStatus == 2 {
			return errors.New("消息已处理过了")
		}

		var content *okr.MsgContentAlignReq
		err = json.Unmarshal([]byte(msg.Content), &content)
		if err != nil {
			return fmt.Errorf("unmarshal msg.Content: %w", err)
		}
		content.MsgContentAlign.ReqHandleBy = msg.ToUser
		if in.Status == 3 {
			content.MsgContentAlign.ReqHandleResult = 1
		} else {
			content.MsgContentAlign.ReqHandleResult = 2
		}

		contentStr, _ := json.Marshal(content)
		msg.Content = string(contentStr)
		msg.ApprovalStatus = 2

		err = okr.NewXOkrMsgModel(sqlx.NewSqlConnFromSession(session)).Update(ctx, msg)
		if err != nil {
			return fmt.Errorf("NewXOkrMsgModel.Update: %w", err)
		}

		period, err := okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, content.EntityPath[0].PeriodId)
		if err != nil {
			rtnErr = fmt.Errorf("NewXOkrPeriodModel.FindOne: %w", err)
			return nil
		}
		// 已经确认过的OKR不允许修改
		if period.ApprovalStatus == int64(oa.OkrStatus_OKR_STATUS_ACKED) {
			rtnErr = fmt.Errorf("用户 %s 的okr已确认", period.Owner)
			return nil
		}
		period2, err := okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, content.AlignWithEntityEntityPath[0].PeriodId)
		if err != nil {
			rtnErr = fmt.Errorf("NewXOkrPeriodModel.FindOne: %w", err)
			return nil
		}
		// 已经确认过的OKR不允许修改
		if period2.ApprovalStatus == int64(oa.OkrStatus_OKR_STATUS_ACKED) {
			rtnErr = fmt.Errorf("用户 %s 的okr已确认", period2.Owner)
			return nil
		}

		if in.Status == 3 {
			alignment, err := okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).FindOneByEntityIdAlignWithEntity(ctx, content.EntityId, content.AlignWithEntityId)
			if err != nil {
				if errors.Is(err, sqlx.ErrNotFound) {
					return nil
				}
				return fmt.Errorf("NewXOkrAlignmentModel.FindOneByEntityIdAlignWithEntity: %w", err)
			}
			err = okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).Delete(ctx, alignment.Id)
			if err != nil {
				return fmt.Errorf("NewXOkrAlignmentModel.Delete: %w", err)
			}
		} else {
			alignment, err := okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).FindOneByEntityIdAlignWithEntity(ctx, content.EntityId, content.AlignWithEntityId)
			if err != nil {
				if errors.Is(err, sqlx.ErrNotFound) {
					rtnErr = errors.New("对齐关系不存在")
					return nil
				}
				return fmt.Errorf("NewXOkrAlignmentModel.FindOneByEntityIdAlignWithEntity: %w", err)
			}
			alignment.Intended = 0
			err = okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).Update(ctx, alignment)
			if err != nil {
				return fmt.Errorf("NewXOkrAlignmentModel.Update: %w", err)
			}
		}
		alignRespContent := &okr.MsgContentAlignResp{
			MsgContentAlign: content.MsgContentAlign,
		}
		alignRespContentStr, _ := json.Marshal(alignRespContent)

		respMsg := &okr.XOkrMsg{
			Id:             0,
			FromUser:       msg.ToUser,
			ToUser:         msg.FromUser,
			Type:           2,
			Content:        string(alignRespContentStr),
			ApprovalStatus: 1,
			StartDate:      msg.StartDate,
			EndDate:        msg.EndDate,
			CreatedAt:      time.Time{},
			UpdatedAt:      time.Time{},
		}
		_, err = okr.NewXOkrMsgModel(sqlx.NewSqlConnFromSession(session)).Insert(ctx, respMsg)
		if err != nil {
			return fmt.Errorf("NewXOkrMsgModel.Insert: %w", err)
		}

		return nil
	})
	if txErr != nil {
		return nil, txErr
	}
	if rtnErr != nil {
		return nil, rtnErr
	}
	return &oa.EmptyResponse{}, nil
}
