package okrrpclogic

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"oa-server/app/oacenter/model/okr"
	"time"

	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type MessageAssignHandleLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewMessageAssignHandleLogic(ctx context.Context, svcCtx *svc.ServiceContext) *MessageAssignHandleLogic {
	return &MessageAssignHandleLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *MessageAssignHandleLogic) MessageAssignHandle(in *oa.MessageAssignHandleReq) (*oa.EmptyResponse, error) {

	var rtnErr error

	txErr := l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, session sqlx.Session) error {
		msg, err := okr.NewXOkrMsgModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, in.MessageId)
		if err != nil {
			return fmt.Errorf("NewXOkrMsgModel.FindOne: %w", err)
		}
		if msg.Type != 3 {
			return errors.New("这不是一个分配请求消息")
		}
		if msg.ToUser != in.UserEmail {
			return errors.New("你没有处理权限")
		}
		if in.Status == 3 { //拒绝
			//do nothing
		} else if in.Status == 4 { //接受为O
			//do nothing
		} else if in.Status == 5 { //接受为KR
			//do nothing
		} else {
			return errors.New("unknown status")
		}
		if msg.ApprovalStatus == 2 {
			return errors.New("消息已处理过了")
		}

		var content *okr.MsgContentAssignReq
		err = json.Unmarshal([]byte(msg.Content), &content)
		if err != nil {
			return fmt.Errorf("unmarshal msg.Content: %w", err)
		}
		content.MsgContentAssign.ReqHandleBy = msg.ToUser
		if in.Status == 3 {
			content.MsgContentAssign.ReqHandleResult = 1
		} else {
			content.MsgContentAssign.ReqHandleResult = 2
		}

		contentStr, _ := json.Marshal(content)
		msg.Content = string(contentStr)
		msg.ApprovalStatus = 2

		err = okr.NewXOkrMsgModel(sqlx.NewSqlConnFromSession(session)).Update(ctx, msg)
		if err != nil {
			return fmt.Errorf("NewXOkrMsgModel.Update: %w", err)
		}

		entity, err := okr.NewXOKrTaskModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, content.EntityId)
		if err != nil {
			rtnErr = errors.New("条目不存在")
			return nil
		}
		if entity.Owner != msg.FromUser {
			rtnErr = fmt.Errorf("条目所有权已不属于 %s", msg.FromUser)
			return nil
		}
		if entity.IntendedOwner != msg.ToUser {
			rtnErr = errors.New("消息有了新的分配去向")
			return nil
		}
		period, err := okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, entity.PeriodId)
		if err != nil {
			return fmt.Errorf("NewXOkrPeriodModel.FindOne: %w", err)
		}
		newPeriod, err := okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOneByOwnerStartDateEndDate(ctx, entity.IntendedOwner, period.StartDate, period.EndDate)
		if err != nil {
			if errors.Is(err, sqlx.ErrNotFound) {
				rtnErr = fmt.Errorf(
					"找不到用户 %s 的okr周期 [%s, %s]",
					entity.IntendedOwner,
					period.StartDate.Local().Format(time.DateOnly),
					period.EndDate.Local().Format(time.DateOnly))
				return nil
			}
			return fmt.Errorf("NewXOkrPeriodModel.FindOneByOwnerStartDateEndDate: %w", err)
		}
		// 已经确认过的OKR不允许修改
		if period.ApprovalStatus == int64(oa.OkrStatus_OKR_STATUS_ACKED) {
			rtnErr = fmt.Errorf("用户 %s 的okr 已确认", period.Owner)
			return nil
		}
		// 已经确认过的OKR不允许修改
		if newPeriod.ApprovalStatus == int64(oa.OkrStatus_OKR_STATUS_ACKED) {
			rtnErr = fmt.Errorf("用户 %s 的okr 已确认", newPeriod.Owner)
			return nil
		}

		if in.Status == 3 { //拒绝分配
			entity.IntendedOwner = ""
			err := okr.NewXOKrTaskModel(sqlx.NewSqlConnFromSession(session)).Update(ctx, entity)
			if err != nil {
				return fmt.Errorf("NewXOKrTaskModel.Update: %w", err)
			}
		} else { //接受分配
			alignment := &okr.XOkrAlignment{
				Id:                0,
				EntityId:          entity.EntityId,
				AlignWithEntity:   entity.ParentEntityId,
				AlignWithUser:     entity.Owner,
				AlignWithPosition: entity.Position,
				AlignWithWeight:   entity.Weight,
				Intended:          0,
				ByAssign:          1,
				CreatedAt:         time.Time{},
				UpdatedAt:         time.Time{},
			}

			_, err = okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).Insert(ctx, alignment)
			if err != nil {
				return fmt.Errorf("NewXOkrAlignmentModel.Insert: %w", err)
			}

			entity.PeriodId = newPeriod.PeriodId
			entity.Owner = entity.IntendedOwner
			entity.Position = 0
			entity.Weight = 0
			entity.IntendedOwner = ""
			if in.Status == 4 {
				entity.EntityType = 1
				entity.ParentEntityId = 0
			} else if in.Status == 5 {
				entity.EntityType = 2
				entity.ParentEntityId = in.KrUnderObjectiveId
			}
			if entity.TypeForKr == 0 {
				entity.TypeForKr = 1
			}

			err = okr.NewXOKrTaskModel(sqlx.NewSqlConnFromSession(session)).Update(ctx, entity)
			if err != nil {
				return fmt.Errorf("NewXOKrTaskModel.Update: %w", err)
			}
			// 一并修改冗余的align_with_user列
			_, err = okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).BulkSetAlignWithUser(ctx, session, entity.EntityId, entity.Owner)
			if err != nil {
				return fmt.Errorf("NewXOkrAlignmentModel.BulkSetAlignWithUser: %w", err)
			}
		}
		assignRespContent := &okr.MsgContentAssignResp{
			MsgContentAssign: content.MsgContentAssign,
		}
		assignRespContentStr, _ := json.Marshal(assignRespContent)

		respMsg := &okr.XOkrMsg{
			Id:             0,
			FromUser:       msg.ToUser,
			ToUser:         msg.FromUser,
			Type:           4,
			Content:        string(assignRespContentStr),
			ApprovalStatus: 1,
			StartDate:      msg.StartDate,
			EndDate:        msg.EndDate,
			CreatedAt:      time.Time{},
			UpdatedAt:      time.Time{},
		}
		_, err = okr.NewXOkrMsgModel(sqlx.NewSqlConnFromSession(session)).Insert(ctx, respMsg)
		if err != nil {
			return fmt.Errorf("NewXOkrMsgModel.Insert: %w", err)
		}

		return nil
	})
	if txErr != nil {
		return nil, txErr
	}
	if rtnErr != nil {
		return nil, rtnErr
	}
	return &oa.EmptyResponse{}, nil
}
