package okrrpclogic

import (
	"context"
	"errors"
	"fmt"
	"maps"
	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"slices"
	"strconv"
	"time"

	"github.com/zeromicro/go-zero/core/stores/sqlx"

	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type OkrDetailLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewOkrDetailLogic(ctx context.Context, svcCtx *svc.ServiceContext) *OkrDetailLogic {
	return &OkrDetailLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

var (
	O_KR_TASK_ALL_FIELDS                                                   = []string{"type_for_kr", "content", "weight", "priority", "deadline", "owner"} //case 1
	O_KR_TASK_OWNER_CAN_EDIT_ASSIGNED_ENTITY_FIELDS                        = []string{"type_for_kr", "content", "weight"}                                  //case 2
	O_KR_TASK_CREATOR_CAN_EDIT_ASSIGNED_ENTITY_FIELDS                      = []string{"type_for_kr", "content", "weight", "priority", "deadline"}          //case 4.1
	O_KR_TASK_PASSIVE_ALIGNMENT_LEADER_CAN_EDIT_NON_ASSIGNED_ENTITY_FIELDS = []string{"type_for_kr", "content", "weight", "priority", "deadline"}          //case 3
	O_KR_TASK_PASSIVE_ALIGNMENT_LEADER_CAN_EDIT_ASSIGNED_ENTITY_FIELDS     = []string{"type_for_kr", "content", "weight", "priority", "deadline"}          //case 4.2
)

type EntityId = int64

type AlignmentEntityIdPair struct {
	EntityId          int64
	AlignWithEntityId int64
}

type OKRTreeNode struct {
	EntityRow      *okr.XOKrTask
	DirectChildren map[EntityId]*OKRTreeNode
	AlignChildren  map[AlignmentEntityIdPair]*OKRTreeNode
	AlignParents   map[AlignmentEntityIdPair]*OKRTreeNode
}

func (l *OkrDetailLogic) OkrDetail(in *oa.OkrDetailReq) (*oa.OkrDetailResp, error) {
	startDateStr, err := time.ParseInLocation(time.DateOnly, in.StartDate, time.Local)
	if err != nil {
		return nil, errors.New("invalid start date")
	}
	endDateStr, err := time.ParseInLocation(time.DateOnly, in.EndDate, time.Local)
	if err != nil {
		return nil, errors.New("invalid end date")
	}
	unHandledCount, err := l.svcCtx.OkrMsgModel.UnHandledMessage(&okr.MsgListReq{
		StartDate: startDateStr,
		EndDate:   endDateStr,
		ToUser:    in.SignedInUserEmail,
	})
	if err != nil {
		logx.Errorw("UnReadMessage", logx.Field("error", err), logx.Field("startDate", startDateStr), logx.Field("endDate", endDateStr), logx.Field("userEmail", in.SignedInUserEmail))
		return nil, err
	}

	var period *okr.XOkrPeriod
	var ownerTreeEntities []*okr.XOKrTask
	var alignmentsToHigher []*okr.XOkrAlignment
	var alignWithWhomAndTheirAncestors []*okr.XOKrTask
	var alignmentsFromLower []*okr.XOkrAlignment
	var whoAlignWithMe []*okr.XOKrTask
	var alignmentsByAssign []*okr.XOkrAlignment

	errNoPeriod := errors.New("no period")

	err = l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, session sqlx.Session) error {
		//获取period
		period, err = okr.NewXOkrPeriodModel(sqlx.NewSqlConnFromSession(session)).FindOneByOwnerStartDateEndDate(ctx, in.DataUserEmail, startDateStr, endDateStr)
		if err != nil {
			if errors.Is(err, sqlx.ErrNotFound) {
				return errNoPeriod
			}
		}
		//todo check period 可编辑
		ownerTreeEntities, err = l.svcCtx.OkrTaskModel.GetByPeriod(ctx, session, period.PeriodId)
		if err != nil {
			return fmt.Errorf("OkrTaskModel.GetByPeriod: %w", err)
		}
		entityToEntityId := func(e *okr.XOKrTask) int64 {
			return e.EntityId
		}
		ownerTreeEntityIds := slices.Collect(common.Map(entityToEntityId, slices.Values(ownerTreeEntities)))

		//获取"和哪些节点对齐"（包括我已发出请求但领导还未处理的对齐）
		alignmentsToHigher, err = l.svcCtx.OkrAlignmentModel.GetAlignmentParents(ctx, session, ownerTreeEntityIds, true)
		if err != nil {
			return fmt.Errorf("OkrAlignmentModel.GetAlignmentParents: %w", err)
		}
		alignmentToAlignWithEntityId := func(e *okr.XOkrAlignment) int64 {
			return e.AlignWithEntity
		}
		alignWithWhomEntityIds := slices.Collect(common.Map(alignmentToAlignWithEntityId, slices.Values(alignmentsToHigher)))
		alignWithWhomAndTheirAncestors, err = l.svcCtx.OkrTaskModel.GetEntityRootPaths(ctx, session, alignWithWhomEntityIds)
		if err != nil {
			return fmt.Errorf("OkrTaskModel.GetEntityRootPaths: %w", err)
		}
		alignWithWhomAndTheirAncestorEntityIds := slices.Collect(common.Map(entityToEntityId, slices.Values(alignWithWhomAndTheirAncestors)))

		//获取"哪些节点和我对齐"（不包括别人已发出请求但我还未处理的对齐）
		alignmentsFromLower, err = l.svcCtx.OkrAlignmentModel.GetAlignmentChildren(ctx, session, ownerTreeEntityIds, false)
		if err != nil {
			return fmt.Errorf("OkrAlignmentModel.GetAlignmentChildren: %w", err)
		}
		alignmentToEntityId := func(e *okr.XOkrAlignment) int64 {
			return e.EntityId
		}
		whoAlignWithMeEntityIds := slices.Collect(common.Map(alignmentToEntityId, slices.Values(alignmentsFromLower)))
		whoAlignWithMe, err = l.svcCtx.OkrTaskModel.GetEntities(ctx, session, whoAlignWithMeEntityIds)
		if err != nil {
			return fmt.Errorf("OkrTaskModel.GetEntities: %w", err)
		}

		//获取“这些entity中哪些entity是由assign产生的”
		alignmentsByAssign, err = l.svcCtx.OkrAlignmentModel.GetAlignmentByAssign(ctx, session, slices.Concat(ownerTreeEntityIds, alignWithWhomAndTheirAncestorEntityIds, whoAlignWithMeEntityIds))
		if err != nil {
			return fmt.Errorf("OkrAlignmentModel.GetAlignmentByAssign: %w", err)
		}

		return nil
	})
	if err != nil {
		if errors.Is(err, errNoPeriod) {
			return &oa.OkrDetailResp{
				UnHandledMessages: unHandledCount,
			}, nil
		}
		return nil, err
	}

	//聚合

	//聚合准备
	globalEntityMap := make(map[EntityId]*okr.XOKrTask)
	for _, entity := range slices.Concat(ownerTreeEntities, alignWithWhomAndTheirAncestors, whoAlignWithMe) {
		globalEntityMap[entity.EntityId] = entity
	}
	globalAlignmentMap := make(map[AlignmentEntityIdPair]*okr.XOkrAlignment)
	for _, alignment := range slices.Concat(alignmentsToHigher, alignmentsFromLower) {
		globalAlignmentMap[AlignmentEntityIdPair{
			EntityId:          alignment.EntityId,
			AlignWithEntityId: alignment.AlignWithEntity,
		}] = alignment
	}
	globalAlignmentByAssignMap := make(map[EntityId]*okr.XOkrAlignment)
	for _, alignment := range alignmentsByAssign {
		globalAlignmentByAssignMap[alignment.EntityId] = alignment
	}

	treeNodes := make(map[EntityId]*OKRTreeNode)
	for _, entity := range globalEntityMap {
		treeNodes[entity.EntityId] = &OKRTreeNode{
			EntityRow:      entity,
			DirectChildren: make(map[EntityId]*OKRTreeNode),
			AlignChildren:  make(map[AlignmentEntityIdPair]*OKRTreeNode),
			AlignParents:   make(map[AlignmentEntityIdPair]*OKRTreeNode),
		}
	}
	for _, entity := range globalEntityMap {
		if treeNodes[entity.ParentEntityId] == nil {
			continue
		}
		treeNodes[entity.ParentEntityId].DirectChildren[entity.EntityId] = treeNodes[entity.EntityId]
	}
	for _, alignment := range globalAlignmentMap {
		if treeNodes[alignment.EntityId] == nil {
			continue
		}
		if treeNodes[alignment.AlignWithEntity] == nil {
			continue
		}
		if treeNodes[alignment.AlignWithEntity].EntityRow.Owner == in.DataUserEmail {
			treeNodes[alignment.AlignWithEntity].AlignChildren[AlignmentEntityIdPair{
				EntityId:          alignment.EntityId,
				AlignWithEntityId: alignment.AlignWithEntity,
			}] = treeNodes[alignment.EntityId]
		}
		if treeNodes[alignment.EntityId].EntityRow.Owner == in.DataUserEmail {
			treeNodes[alignment.EntityId].AlignParents[AlignmentEntityIdPair{
				EntityId:          alignment.EntityId,
				AlignWithEntityId: alignment.AlignWithEntity,
			}] = treeNodes[alignment.AlignWithEntity]
		}
	}

	var dataUserTreeRoots []*oa.OKRTreeElem

	var traverse func(node *OKRTreeNode, parentNode *OKRTreeNode) (apiTreeNode *oa.OKRTreeElem)
	traverse = func(node *OKRTreeNode, parentNode *OKRTreeNode) (apiTreeNode *oa.OKRTreeElem) {
		apiTreeNode = &oa.OKRTreeElem{
			Entity:    nil,
			AlignWith: make([]*oa.OKRTreeElem, 0),
			Children:  make([]*oa.OKRTreeElem, 0),
		}
		var alignmentWithParentRowOptional *okr.XOkrAlignment
		if parentNode != nil {
			alignmentWithParentRowOptional = globalAlignmentMap[AlignmentEntityIdPair{
				EntityId:          node.EntityRow.EntityId,
				AlignWithEntityId: parentNode.EntityRow.EntityId,
			}]
		}

		apiTreeNode.Entity = computeOKREntityView(
			node.EntityRow,
			alignmentWithParentRowOptional,
			globalAlignmentByAssignMap[node.EntityRow.EntityId],
			in.DataUserEmail == in.SignedInUserEmail,
		)
		for _, directChild := range node.DirectChildren {
			apiTreeNode.Children = append(apiTreeNode.Children, traverse(directChild, node))
		}
		for _, alignChild := range node.AlignChildren {
			apiTreeNode.Children = append(apiTreeNode.Children, traverse(alignChild, node))
		}
		slices.SortFunc(apiTreeNode.Children, func(a, b *oa.OKRTreeElem) int {
			return int(a.Entity.EntityId - b.Entity.EntityId)
		})

		getPath := func(pathLeaf *OKRTreeNode) (pathRootInApiTreeNodeForm *oa.OKRTreeElem) {
			if pathLeaf == nil {
				//panic("nil elem")
				return nil
			}
			nodeCopy := &oa.OKRTreeElem{
				Entity: computeOKREntityView(
					pathLeaf.EntityRow,
					nil,
					globalAlignmentByAssignMap[pathLeaf.EntityRow.EntityId],
					false,
				),
				AlignWith: nil,
				Children:  nil,
			}
			for {
				parent, ok := treeNodes[nodeCopy.Entity.ParentEntityId]
				if !ok {
					return nodeCopy
				}
				parentCopy := &oa.OKRTreeElem{
					Entity: computeOKREntityView(
						parent.EntityRow,
						nil,
						globalAlignmentByAssignMap[parent.EntityRow.EntityId],
						false,
					),
					AlignWith: nil,
					Children:  []*oa.OKRTreeElem{nodeCopy},
				}
				nodeCopy = parentCopy
			}
		}

		nodeAlignParentsAsSlice := slices.Collect(maps.Values(node.AlignParents))
		slices.SortFunc(nodeAlignParentsAsSlice, func(a, b *OKRTreeNode) int {
			return int(a.EntityRow.EntityId - b.EntityRow.EntityId)
		})
		for _, alignParent := range nodeAlignParentsAsSlice {
			apiTreeNode.AlignWith = append(apiTreeNode.AlignWith, getPath(alignParent))
		}
		return apiTreeNode
	}

	for _, node := range treeNodes {
		if node.EntityRow.Owner == in.DataUserEmail && node.EntityRow.EntityType == 1 {
			apiNode := traverse(node, nil)
			dataUserTreeRoots = append(dataUserTreeRoots, apiNode)
		}
	}
	slices.SortFunc(dataUserTreeRoots, func(a, b *oa.OKRTreeElem) int {
		return int(a.Entity.EntityId - b.Entity.EntityId)
	})

	rtn := &oa.OkrDetailResp{
		OkrId:             period.PeriodId,
		UserEmail:         period.Owner,
		StartDate:         period.StartDate.Local().Format(time.DateOnly),
		EndDate:           period.EndDate.Local().Format(time.DateOnly),
		Status:            period.ApprovalStatus,
		Children:          dataUserTreeRoots,
		UnHandledMessages: unHandledCount,
	}
	return rtn, nil
}

func getEntityAlignmentIncludedChildren(ctx context.Context, session sqlx.Session, entityId int64) (list []*oa.OKREntityView, err error) {
	//获取owner树上的孩子节点
	ownerTreeChildEntities, err := okr.NewXOKrTaskModel(sqlx.NewSqlConnFromSession(session)).GetChildren(entityId)
	if err != nil {
		return nil, err
	}
	entityToEntityId := func(e *okr.XOKrTask) int64 {
		return e.EntityId
	}
	ownerTreeChildEntityIds := slices.Collect(common.Map(entityToEntityId, slices.Values(ownerTreeChildEntities)))

	//获取"哪些节点和我对齐"
	alignmentsFromLower, err := okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).GetAlignmentChildren(ctx, session, ownerTreeChildEntityIds, false)
	if err != nil {
		return nil, err
	}
	alignmentToEntityId := func(e *okr.XOkrAlignment) int64 {
		return e.EntityId
	}
	whoAlignWithMeEntityIds := slices.Collect(common.Map(alignmentToEntityId, slices.Values(alignmentsFromLower)))
	whoAlignWithMe, err := okr.NewXOKrTaskModel(sqlx.NewSqlConnFromSession(session)).GetEntities(ctx, session, whoAlignWithMeEntityIds)
	if err != nil {
		return nil, err
	}

	//获取“哪些entity是由assign产生的”
	alignmentsByAssign, err := okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(session)).GetAlignmentByAssign(
		ctx, session, slices.Concat(ownerTreeChildEntityIds, whoAlignWithMeEntityIds))
	if err != nil {
		return nil, err
	}

	//聚合
	globalAlignmentMap := make(map[AlignmentEntityIdPair]*okr.XOkrAlignment)
	for _, alignment := range alignmentsFromLower {
		globalAlignmentMap[AlignmentEntityIdPair{
			EntityId:          alignment.EntityId,
			AlignWithEntityId: alignment.AlignWithEntity,
		}] = alignment
	}
	globalAlignmentByAssignMap := make(map[EntityId]*okr.XOkrAlignment)
	for _, alignment := range alignmentsByAssign {
		globalAlignmentByAssignMap[alignment.EntityId] = alignment
	}

	for _, entity := range ownerTreeChildEntities {
		list = append(list, computeOKREntityView(
			entity,
			nil,
			globalAlignmentByAssignMap[entity.EntityId],
			false,
		))
	}
	for _, entity := range whoAlignWithMe {
		list = append(list, computeOKREntityView(
			entity,
			globalAlignmentMap[AlignmentEntityIdPair{
				EntityId:          entity.EntityId,
				AlignWithEntityId: entityId,
			}],
			globalAlignmentByAssignMap[entity.EntityId],
			false,
		))
	}
	slices.SortFunc(list, func(a, b *oa.OKREntityView) int {
		return int(a.EntityId - b.EntityId)
	})
	return list, nil
}

func computeOKREntityView(
	entityRow *okr.XOKrTask,
	alignmentWithParentRowOptional *okr.XOkrAlignment,
	alignmentWithAssignerRowOptional *okr.XOkrAlignment,
	needEdition bool,
) *oa.OKREntityView {
	if alignmentWithParentRowOptional != nil && alignmentWithParentRowOptional.EntityId != entityRow.EntityId {
		panic("invalid alignmentWithParentRowOptional")
	}
	if alignmentWithAssignerRowOptional != nil && alignmentWithAssignerRowOptional.EntityId != entityRow.EntityId {
		panic("invalid alignmentWithAssignerRowOptional")
	}
	rtn := &oa.OKREntityView{
		AlignWithParent:         false, //表示在oa.OKRTreeElem树形结构上对parent节点的关系是否是alignment关系
		AlignWithParentByAssign: false, //表示在oa.OKRTreeElem树形结构上对parent节点的alignment关系是否由parent节点领导的assign动作发起
		ByAssign:                false, //表示EntityId记录是不是某个领导的assign动作发起
		EntityId:                entityRow.EntityId,
		EntityType:              entityRow.EntityType,
		ParentEntityId:          entityRow.ParentEntityId,
		Content:                 entityRow.Content,
		Priority:                entityRow.Priority,
		Deadline:                entityRow.Deadline.Local().Format(time.DateOnly),
		TypeForKr:               entityRow.TypeForKr,
		Owner:                   entityRow.Owner,
		Position:                entityRow.Position,
		Weight:                  strconv.FormatFloat(entityRow.Weight, 'f', 2, 64),
		IntendedOwner:           entityRow.IntendedOwner,
		IntendedAlign:           0,
		EditableBySignedInUser:  nil,
	}

	if alignmentWithParentRowOptional != nil {
		rtn.AlignWithParent = true
		if alignmentWithParentRowOptional.ByAssign == 1 {
			rtn.AlignWithParentByAssign = true
		}
		rtn.Position = alignmentWithParentRowOptional.AlignWithPosition
		rtn.Weight = strconv.FormatFloat(alignmentWithParentRowOptional.AlignWithWeight, 'f', 2, 64)
		rtn.IntendedAlign = alignmentWithParentRowOptional.Intended
	}

	if alignmentWithAssignerRowOptional != nil {
		rtn.ByAssign = true
	}

	populateEditable := func(entityView *oa.OKREntityView) {
		if entityView.AlignWithParent { //是alignment关系形成的孩子，owner不是signedInUser, 而是userFoo
			if entityView.ByAssign { // 孩子是某个人分配给userFoo的
				if entityView.AlignWithParentByAssign { //孩子是signedInUser分配给userFoo的
					entityView.EditableBySignedInUser = O_KR_TASK_CREATOR_CAN_EDIT_ASSIGNED_ENTITY_FIELDS
				} else { //孩子是非signedInUser的某个人分配给userFoo的
					entityView.EditableBySignedInUser = O_KR_TASK_PASSIVE_ALIGNMENT_LEADER_CAN_EDIT_ASSIGNED_ENTITY_FIELDS
				}
			} else { // 孩子是userFoo自建的
				entityView.EditableBySignedInUser = O_KR_TASK_PASSIVE_ALIGNMENT_LEADER_CAN_EDIT_NON_ASSIGNED_ENTITY_FIELDS
			}
		} else { // 是signedInUser的真孩子， owner是signedInUser
			if entityView.ByAssign { //孩子某个人分配给signedInUser的
				entityView.EditableBySignedInUser = O_KR_TASK_OWNER_CAN_EDIT_ASSIGNED_ENTITY_FIELDS
			} else { //孩子是signedInUser自建的
				entityView.EditableBySignedInUser = O_KR_TASK_ALL_FIELDS
			}
		}
	}

	if needEdition {
		populateEditable(rtn)
	}

	return rtn
}
