package okrrpclogic

import (
	"context"
	"errors"
	"fmt"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"maps"
	"oa-server/app/oacenter/model/okr"

	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type OkrPosUpdateLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewOkrPosUpdateLogic(ctx context.Context, svcCtx *svc.ServiceContext) *OkrPosUpdateLogic {
	return &OkrPosUpdateLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *OkrPosUpdateLogic) OkrPosUpdate(in *oa.OkrPosUpdateReq) (*oa.EmptyResponse, error) {
	err := l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, session sqlx.Session) error {
		parentEntity, err := okr.NewXOKrTaskModel(sqlx.NewSqlConnFromSession(session)).FindOne(ctx, in.ParentEntityId)
		if err != nil {
			return fmt.Errorf("NewXOKrTaskModel.FindOne: %w", err)
		}
		if parentEntity.Owner != in.UserEmail {
			return errors.New("你没有变更权限")
		}
		childrenEntities, err := getEntityAlignmentIncludedChildren(ctx, session, parentEntity.EntityId)
		if err != nil {
			return fmt.Errorf("getEntityAlignmentIncludedChildren: %w", err)
		}
		allEntityIdsInDB := make(map[int64]struct{})
		for _, childEntity := range childrenEntities {
			allEntityIdsInDB[childEntity.EntityId] = struct{}{}
		}
		allEntityIdsInReq := make(map[int64]struct{})
		for _, id := range in.EntityIds {
			allEntityIdsInReq[id] = struct{}{}
		}
		if !maps.Equal(allEntityIdsInDB, allEntityIdsInReq) {
			return errors.New("列表元素集合和后端数据不一致")
		}

		byOwnership := make(map[int64]int64)
		byAlignment := make(map[int64]int64)
		for _, childEntity := range childrenEntities {
			if childEntity.AlignWithParent {
				byAlignment[childEntity.EntityId] = 0
			} else {
				byOwnership[childEntity.EntityId] = 0
			}
		}
		for newIdx, childEntityId := range in.EntityIds {
			if _, ok := byOwnership[childEntityId]; ok {
				byOwnership[childEntityId] = int64(newIdx)
			}
			if _, ok := byAlignment[childEntityId]; ok {
				byAlignment[childEntityId] = int64(newIdx)
			}
		}

		_, err = l.svcCtx.OkrTaskModel.BulkUpdatePosition(ctx, session, byOwnership)
		if err != nil {
			return fmt.Errorf("OkrTaskModel.BulkUpdatePosition: %w", err)
		}
		_, err = l.svcCtx.OkrAlignmentModel.BulkUpdatePosition(ctx, session, byAlignment)
		if err != nil {
			return fmt.Errorf("OkrAlignmentModel.BulkUpdatePosition: %w", err)
		}

		return nil
	})
	if err != nil {
		return nil, err
	}

	return &oa.EmptyResponse{}, nil
}
