package okrrpclogic

import (
	"context"
	"encoding/json"
	"errors"
	"time"

	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type OkrTaskAddLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewOkrTaskAddLogic(ctx context.Context, svcCtx *svc.ServiceContext) *OkrTaskAddLogic {
	return &OkrTaskAddLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}
func checkOkrStatus(model okr.XOkrPeriodModel, okrID int64) (*okr.XOkrPeriod, error) {
	okrPeriod, err := model.FindOne(context.Background(), okrID)
	if err != nil {
		logx.Errorw("未找到对应OKR", logx.Field("okr_id", okrID), logx.Field("err", err))
		return nil, err
	}
	if okrPeriod.ApprovalStatus == 3 {
		return nil, errors.New("OKR已确认，无法修改")
	}
	if isCreateOkr(okrPeriod.StartDate.Format(time.DateOnly)) {
		return okrPeriod, nil
	} else {
		return nil, ErrOkrPeriodInvalid
	}
}
func (l *OkrTaskAddLogic) OkrTaskAdd(in *oa.OkrTaskAddReq) (*oa.EmptyResponse, error) {
	okrPeriod, err := checkOkrStatus(l.svcCtx.OkrPeriodModel, int64(in.OkrId))
	if err != nil {
		return nil, err
	}
	if okrPeriod.Owner != in.Operator {
		return nil, ErrPermissionDenied
	}

	deadline, err := time.ParseInLocation(time.DateOnly, in.Deadline, time.Local)
	if err != nil {
		logx.Errorw("截止时间格式错误", logx.Field("err", err), logx.Field("deadline", in.Deadline))
		return nil, err
	}
	taskListModel, err := l.svcCtx.OkrTaskModel.GetChildren(int64(in.KeyresultId))
	if err != nil {
		logx.Errorw("未找到对应KR", logx.Field("kr_id", in.KeyresultId), logx.Field("err", err))
		return nil, err
	}
	okrTask := &okr.XOKrTask{
		EntityType:     3,
		ParentEntityId: int64(in.KeyresultId),
		PeriodId:       int64(in.OkrId),
		Content:        in.Content,
		Priority:       in.Priority,
		Deadline:       deadline,
		Owner:          in.Operator,
		Position:       int64(len(taskListModel) + 1),
		Weight:         float64(in.Weight),
	}
	if in.Operator != in.UserEmail {
		okrTask.IntendedOwner = in.UserEmail
	}

	err = l.addTask(okrTask, okrPeriod, in)
	return &oa.EmptyResponse{}, err
}

func (l *OkrTaskAddLogic) addTask(okrTask *okr.XOKrTask, okrPeriod *okr.XOkrPeriod, in *oa.OkrTaskAddReq) error {

	conn := sqlx.NewMysql(l.svcCtx.Config.DB.DataSource)
	return conn.Transact(func(session sqlx.Session) error {
		result, err := l.svcCtx.OkrTaskModel.InsertWithSession(session, okrTask)
		if err != nil {
			logx.Errorw("新增task失败", logx.Field("err", err), logx.Field("task_model", okrTask))
			return err
		}
		// 如果是指派任务，添加消息
		if in.Operator != in.UserEmail {
			// 获取最后插入的ID
			taskID, err := result.LastInsertId()
			if err != nil {
				logx.Errorw("获取task ID失败", logx.Field("err", err))
				return err
			}

			req := &oa.OkrTaskUpdateReq{
				TaskId: uint64(taskID),
				AddReq: in,
			}
			if err := addAssignMsgWithTx(l.svcCtx.OkrMsgModel, l.svcCtx.OkrTaskModel, session, okrPeriod, okrTask, req); err != nil {
				return err
			}
		}
		return nil
	})
}

func addAssignMsgWithTx(okrMsgModel okr.XOkrMsgModel, taskModel okr.XOKrTaskModel, session sqlx.Session, okrPeriod *okr.XOkrPeriod, okrTask *okr.XOKrTask, req *oa.OkrTaskUpdateReq) error {
	okrO, err := taskModel.FindOne(context.Background(), int64(req.AddReq.ObjectiveId))
	if err != nil {
		logx.Errorw("获取o详情失败", logx.Field("err", err), logx.Field("o_id", req.AddReq.ObjectiveId))
		return err
	}
	okrKR, err := taskModel.FindOne(context.Background(), int64(req.AddReq.KeyresultId))
	if err != nil {
		logx.Errorw("获取kr详情失败", logx.Field("err", err), logx.Field("kr_id", req.AddReq.KeyresultId))
		return err
	}
	msgContent := &okr.MsgContentAssignReq{
		MsgContentAssign: okr.MsgContentAssign{
			EntityId:    int64(req.TaskId),
			ReqBy:       req.AddReq.Operator,
			ReqHandleBy: req.AddReq.UserEmail,
			EntityPath:  []*okr.XOKrTask{okrO, okrKR, okrTask},
		},
	}
	b, err := json.Marshal(msgContent)
	if err != nil {
		logx.Errorw("消息内容序列化失败", logx.Field("err", err))
		return err
	}

	msgModel := &okr.XOkrMsg{
		FromUser:       req.AddReq.Operator,
		ToUser:         req.AddReq.UserEmail,
		Type:           3, // 1:对齐Req 2:对齐Resp 3:分配Req 4:分配Resp 5:确认Req 6:确认Resp
		Content:        string(b),
		ApprovalStatus: 1,
		StartDate:      okrPeriod.StartDate,
		EndDate:        okrPeriod.EndDate,
	}

	_, err = okrMsgModel.InsertWithSession(session, msgModel)
	if err != nil {
		logx.Errorw("发送消息失败", logx.Field("err", err), logx.Field("task id", req.TaskId))
		return err
	}
	return nil
}
