package okrrpclogic

import (
	"context"
	"database/sql"
	"errors"
	"time"

	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type OkrTaskUpdateLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewOkrTaskUpdateLogic(ctx context.Context, svcCtx *svc.ServiceContext) *OkrTaskUpdateLogic {
	return &OkrTaskUpdateLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// OkrTaskUpdate 这里只能是创建人修改
// 被指派人接受指派后，生成的是O或KR，不会生成task
func (l *OkrTaskUpdateLogic) OkrTaskUpdate(in *oa.OkrTaskUpdateReq) (*oa.EmptyResponse, error) {
	okrPeriod, err := checkOkrStatus(l.svcCtx.OkrPeriodModel, int64(in.AddReq.OkrId))
	if err != nil {
		return nil, err
	}

	if okrPeriod.Owner != in.AddReq.Operator {
		return nil, ErrPermissionDenied
	}
	// 获取okrTask
	okrTask, err := l.svcCtx.OkrTaskModel.FindOne(l.ctx, int64(in.TaskId))
	if err != nil {
		if err == sql.ErrNoRows {
			return nil, errors.New("未找到对应任务")
		}
		logx.Errorw("获取task详情失败", logx.Field("id", in.TaskId))
		return nil, err
	}
	// 不是owner时，不能修改指派人
	if okrTask.Owner != in.AddReq.Operator && okrTask.Owner != in.AddReq.UserEmail {
		return nil, errors.New("您非当前task的owner，无权限修改指派人")
	}
	deadline, err := time.ParseInLocation(time.DateOnly, in.AddReq.Deadline, time.Local)
	if err != nil {
		logx.Errorw("截止时间格式错误", logx.Field("err", err), logx.Field("deadline", in.AddReq.Deadline))
		return nil, err
	}
	okrTask.Content = in.AddReq.Content
	okrTask.Priority = in.AddReq.Priority
	okrTask.Deadline = deadline
	if okrTask.Owner == in.AddReq.Operator {
		okrTask.Weight = float64(in.AddReq.Weight)
	}
	err = l.updateTask(okrPeriod, okrTask, in)
	if err != nil {
		return nil, err
	}

	return &oa.EmptyResponse{}, nil
}

func (l *OkrTaskUpdateLogic) updateTask(okrPeriod *okr.XOkrPeriod, okrTask *okr.XOKrTask, in *oa.OkrTaskUpdateReq) error {
	conn := sqlx.NewMysql(l.svcCtx.Config.DB.DataSource)
	return conn.Transact(func(session sqlx.Session) error {
		// 修改指派人的情况
		oldIntendedOwner := okrTask.IntendedOwner
		taskState := checkTaskState(in.AddReq.UserEmail, in.AddReq.Operator, oldIntendedOwner)
		okrTask.IntendedOwner = taskState.IntendedOwner
		_, err := l.svcCtx.OkrTaskModel.UpdateWithSession(session, okrTask)
		if err != nil {
			logx.Errorw("更新task失败", logx.Field("err", err), logx.Field("task_model", okrTask))
			return err
		}
		alignments, err := l.svcCtx.OkrAlignmentModel.GetAlignWithEntityIdAndAlignID(l.ctx, okrTask.EntityId, int64(in.AddReq.KeyresultId))
		if err != nil {
			if err != sql.ErrNoRows {
				logx.Errorw("获取关联关系失败", logx.Field("id", okrTask.EntityId))
				return err
			}
		}
		if len(alignments) != 0 {
			if taskState.NeedDeleteRelation {
				alignmentsID := []int64{}
				for _, v := range alignments {
					alignmentsID = append(alignmentsID, v.Id)
				}
				err := l.svcCtx.OkrAlignmentModel.TransDelete(session, alignmentsID)
				if err != nil {
					logx.Errorw("删除关联关系失败", logx.Field("err", err), logx.Field("task_id", okrTask.EntityId))
					return err

				}
			} else {
				// 不需要删除关联关系，但是存在关联关系时，更新权重
				for _, alignment := range alignments {
					alignment.AlignWithWeight = float64(in.AddReq.Weight)
					err = l.svcCtx.OkrAlignmentModel.TransUpdateCtx(l.ctx, session, alignment)
					if err != nil {
						logx.Errorw("更新关联关系权重失败", logx.Field("task id", alignment.EntityId), logx.Field("align id", alignment.AlignWithEntity))
						return err
					}
				}
			}
		}
		if taskState.NeedMsg {
			if err := addAssignMsgWithTx(l.svcCtx.OkrMsgModel, l.svcCtx.OkrTaskModel, session, okrPeriod, okrTask, in); err != nil {
				return err
			}
		}
		return nil
	})

}

type TaskState struct {
	IntendedOwner      string // 指派人
	NeedMsg            bool   // 是否需要发送消息
	NeedDeleteRelation bool   // 是否需要删除关联关系
}

func checkTaskState(currentAssignee, operator, oldIntendedOwner string) TaskState {
	// 1. 从他人修改成自己，不需要发消息，需要删除关联关系
	if oldIntendedOwner != "" && currentAssignee == operator {
		return TaskState{
			IntendedOwner:      "",
			NeedMsg:            false,
			NeedDeleteRelation: true,
		}
	}
	// 2. 从自己修改成他人，需要发消息，不需要删除关联关系
	if oldIntendedOwner == "" && currentAssignee != operator {
		return TaskState{
			IntendedOwner:      currentAssignee,
			NeedMsg:            true,
			NeedDeleteRelation: false,
		}
	}
	// 3. 从A修改成B，需要发消息，需要删除关联关系
	if oldIntendedOwner != "" && oldIntendedOwner != currentAssignee {
		return TaskState{
			IntendedOwner:      oldIntendedOwner,
			NeedMsg:            true,
			NeedDeleteRelation: true,
		}
	}
	return TaskState{
		IntendedOwner:      oldIntendedOwner,
		NeedMsg:            false,
		NeedDeleteRelation: false,
	}
}
