package okrrpclogic

import (
	"context"
	"encoding/json"
	"errors"
	"time"

	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/tool"

	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type UpdateKeyResultLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewUpdateKeyResultLogic(ctx context.Context, svcCtx *svc.ServiceContext) *UpdateKeyResultLogic {
	return &UpdateKeyResultLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 更新KR
func (l *UpdateKeyResultLogic) UpdateKeyResult(in *oa.UpdateKeyResultReq) (*oa.UpdateKeyResultResp, error) {
	deadline, err := time.Parse(time.DateOnly, in.GetDeadline())
	if err != nil {
		return nil, status.Error(codes.InvalidArgument, "无效的deadline")
	}

	myOkr, err := l.svcCtx.OkrPeriodModel.FindOne(l.ctx, in.GetOkrId())
	if err != nil {
		if errors.Is(err, okr.ErrNotFound) {
			return nil, ErrOkrNotFound
		}
		logx.Errorw("failed to query okr_period on UpdateKeyResult", logx.Field("error", err))
		return nil, ErrOkrPeriodQueryFailed
	}
	if !IsOkrEditable(myOkr, in.GetOwner()) {
		logx.Infow("This okr is not editable", logx.Field("period_id", myOkr.PeriodId))
		return nil, ErrPermissionDenied
	}

	objectiveList, err := l.svcCtx.OkrTaskModel.GetByOkrPeriodIdAndEntityType(myOkr.PeriodId, OkrEntityTypeObjective)
	if err != nil {
		logx.Errorw("failed to query objective on UpdateKeyResult", logx.Field("error", err))
		return nil, ErrOkrTaskQueryFailed
	}
	myObjective := objectiveList.GetByEntityId(in.GetObjectiveId())
	if myObjective == nil {
		logx.Infow("this objective doesn't belongs to you", logx.Field("period_id", myOkr.PeriodId), logx.Field("objective_id", in.GetObjectiveId()))
		return nil, ErrPermissionDenied
	}

	// 当前用户的KR来源有三种
	// 1. 自建；可修改所有字段；
	// 2. 其他人分配的；intended_owner,priority,deadline 不可修改；
	//		1,2中的KR是可以直接在x_o_kr_task中通过period_id查询出来的。
	// 3. 其他人对齐的；intended_owner,不可修改。注意weight需要在align表中修改。
	isAlignedKR := false
	isAssignedKR := false
	kr, err := l.svcCtx.OkrTaskModel.GetOneUserOkrEntity(l.ctx, in.GetOkrId(), in.GetObjectiveId(), in.GetKeyresultId())
	if err != nil && !errors.Is(err, okr.ErrNotFound) {
		logx.Errorw("failed to query keyresult on UpdateKeyResult", logx.Field("error", err))
		return nil, ErrOkrTaskQueryFailed
	}
	if kr == nil {
		// 说明更新的是来自对齐的KR
		alignedKrList, err := l.svcCtx.OkrTaskModel.FindByUserAndAlignWithEntity(l.ctx, in.GetOwner(), in.GetObjectiveId())
		if err != nil {
			logx.Errorw("failed to query alignedKeyResultList on UpdateKeyResult", logx.Field("error", err))
			return nil, ErrOkrTaskQueryFailed
		}
		kr = alignedKrList.GetByEntityId(in.GetKeyresultId())
		if kr == nil {
			logx.Errorw("the sepecified keyresult_id is not found on UpdateKeyresult", logx.Field("keyresult_id", in.GetKeyresultId()))
			return nil, ErrOkrTaskNotFound
		}
		isAlignedKR = true
	} else {
		// 查询当前KR是否有对齐其他entity，如果有，则说明当前KR是被分配给自己的，不是自建的。
		alignRecord, err := l.svcCtx.OkrAlignmentModel.GetByEntityId(l.ctx, in.GetKeyresultId())
		if err != nil && !errors.Is(err, okr.ErrNotFound) {
			logx.Errorw("failed to query x_okr_alignment on UpdateKeyresult", logx.Field("error", err), logx.Field("keyresult_id", in.GetKeyresultId()))
		}
		if alignRecord != nil {
			isAssignedKR = true
		}
	}

	// 当前KR是被分配的，则不能再分配给其他人
	if isAssignedKR {
		if in.GetUserEmail() != in.GetOwner() {
			return nil, status.Error(codes.InvalidArgument, "指派给你的keyresult无法再指派给其他人")
		}
		if kr.Priority != in.GetPriority() {
			return nil, status.Error(codes.InvalidArgument, "无法修改被指派keyresult的优先级")
		}
		if !tool.IsSameDay(kr.Deadline, deadline) {
			return nil, status.Error(codes.InvalidArgument, "无法修改被指派keyresult的截止日期")
		}
	}

	// 当前KR是别人对齐来的，则不能修改当前KR的owner
	if isAlignedKR && in.GetUserEmail() != kr.Owner {
		return nil, status.Error(codes.InvalidArgument, "无法修改来自对齐的keyresult的负责人")
	}

	err = l.svcCtx.OkrTaskModel.TransCtx(l.ctx, func(ctx context.Context, s sqlx.Session) error {
		// 修改通用字段
		kr.Content = in.GetContent()
		kr.TypeForKr = int64(in.GetTypeForKr())

		// 自建的KR, 可以修改所有字段
		if !isAlignedKR && !isAssignedKR {
			kr.Priority = in.GetPriority()
			kr.Deadline = deadline
			kr.Weight = in.GetWeight()
			kr.IntendedOwner = ""
			// 指派给其他人
			if in.GetOwner() != in.GetUserEmail() {
				kr.IntendedOwner = in.GetUserEmail()

				// 发消息
				msgContent := okr.MsgContentAssign{
					EntityId:   kr.EntityId,
					ReqBy:      in.GetOwner(),
					EntityPath: []*okr.XOKrTask{myObjective, kr},
				}
				msgContentBytes, _ := json.Marshal(&msgContent)
				msg := okr.XOkrMsg{
					FromUser:       in.GetOwner(),
					ToUser:         in.GetUserEmail(),
					Type:           OkrMessgeTypeAssignReq,
					Content:        string(msgContentBytes),
					ApprovalStatus: OkrMessageApprovalStatusUnhandled,
					StartDate:      myOkr.StartDate,
					EndDate:        myOkr.EndDate,
				}
				msgInsertResult, err := l.svcCtx.OkrMsgModel.InsertWithSession(s, &msg)
				if err != nil {
					logx.Errorw("failed to create okr_message on UpdateeKeyResult", logx.Field("error", err))
					return err
				}
				_, err = msgInsertResult.LastInsertId()
				if err != nil {
					logx.Errorw("failed to get okr_message.LastInsertId on UpdateKeyResult", logx.Field("error", err))
					return err
				}
			}
		}

		// 来自分配的KR
		if isAssignedKR {
			kr.Weight = in.GetWeight()
		}

		// 来自对齐的KR,更新对齐表中的weight
		if isAlignedKR {
			kr.Priority = in.GetPriority()
			kr.Deadline = deadline

			alignRecord, err := l.svcCtx.OkrAlignmentModel.TransGetOneByUserAndAlignWithEntityId(ctx, s, in.GetOwner(), in.GetObjectiveId(), in.GetKeyresultId())
			if err != nil {
				logx.Errorw("failed to get keyresult alignment on UpdateKeyResult", logx.Field("error", err))
				return err
			}
			alignRecord.AlignWithWeight = in.GetWeight()

			if err = okr.NewXOkrAlignmentModel(sqlx.NewSqlConnFromSession(s)).Update(ctx, alignRecord); err != nil {
				logx.Errorw("failed to update x_okr_alignment on UpdateKeyResult", logx.Field("error", err))
				return err
			}
		}

		// 更新keyresult
		if err = okr.NewXOKrTaskModel(sqlx.NewSqlConnFromSession(s)).Update(ctx, kr); err != nil {
			logx.Errorw("failed to update x_o_kr_task on UpdateKeyResult", logx.Field("error", err))
			return err
		}

		return nil
	})
	if err != nil {
		return nil, ErrOkrTaskUpdateFailed
	}

	return &oa.UpdateKeyResultResp{}, nil
}
