package performancerpclogic

import (
	"fmt"
	"strings"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/shopspring/decimal"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

const (
	bonusMonthBase = 2000 // 每月绩效奖金基数

	maxPerformanceItemCommentDisplayCnt = 5

	PerformanceLevelA           = "A"
	PerformanceLevelAFactor     = 1.4
	PerformanceLevelBPlus       = "B+"
	PerformanceLevelBPlusFactor = 1.2
	PerformanceLevelB           = "B"
	PerformanceLevelBFactor     = 1.0
	PerformanceLevelC           = "C"
	PerformanceLevelCFactor     = 0.8
	PerformanceLevelD           = "D"
	PerformanceLevelDFactor     = 0.6
	PerformanceLevelE           = "E"
	PerformanceLevelEFactor     = 0

	additionItemCoworker = "其他部门同事评价"
)

var (
	percent                            = decimal.NewFromInt(100)
	minPerformanceScore        float64 = 0
	maxPerformanceScore        float64 = 150
	staffWorkWeightTotal       float64 = 75
	staffAdditionWeightTotal   float64 = 25
	managerWorkWeightTotal     float64 = 70
	managerAdditionWeightTotal float64 = 30
)

var (
	ErrInvalidParams    = status.Error(codes.InvalidArgument, "无效的参数")
	ErrPermissionDenied = status.Error(codes.PermissionDenied, "无权限操作")

	// 绩效
	ErrPerformanceNotFound                = status.Error(codes.NotFound, "当前绩效未找到")
	ErrPerformanceCreateFailed            = status.Error(codes.Internal, "创建绩效失败")
	ErrPerformanceQueryFailed             = status.Error(codes.Internal, "查询绩效失败")
	ErrPerformanceUpdateFailed            = status.Error(codes.Internal, "更新绩效失败")
	ErrPerformanceCanNotBeModified        = status.Error(codes.PermissionDenied, "当前绩效不允许修改")
	ErrPerformanceNewEmployeeCreateFailed = status.Error(codes.Internal, "新入职员工无需创建绩效")

	// 绩效附加项
	ErrPerformanceAdditionItemNotFound     = status.Error(codes.NotFound, "当前绩效附加项未找到")
	ErrPerformanceAdditionItemCreateFailed = status.Error(codes.Internal, "创建绩效附加项失败")
	ErrPerformanceAdditionItemQueryFailed  = status.Error(codes.Internal, "查询绩效附加项失败")
	ErrPerformanceAdditionItemUpdateFailed = status.Error(codes.Internal, "更新绩效附加项失败")
	ErrPerformanceAdditionItemDeleteFailed = status.Error(codes.Internal, "更新绩效附加项失败")

	// 绩效工作项
	ErrPerformanceWorkItemNotFound     = status.Error(codes.NotFound, "当前绩效工作项未找到")
	ErrPerformanceWorkItemCreateFailed = status.Error(codes.NotFound, "创建绩效工作项失败")
	ErrPerformanceWorkItemQueryFailed  = status.Error(codes.NotFound, "查询绩效工作项失败")
	ErrPerformanceWorkItemUpdateFailed = status.Error(codes.NotFound, "更新绩效工作项失败")
	ErrPerformanceWorkItemDeleteFailed = status.Error(codes.NotFound, "删除绩效工作项失败")

	// 绩效评论
	ErrPerformanceItemCommentNotFound     = status.Error(codes.NotFound, "当前绩效的评论未找到")
	ErrPerformanceItemCommentCreateFailed = status.Error(codes.Internal, "创建绩效评论失败")
	ErrPerformanceItemCommentQueryFailed  = status.Error(codes.Internal, "查询绩效评论失败")
	ErrPerformanceItemCommentUpdateFailed = status.Error(codes.Internal, "更新绩效评论失败")
	ErrPerformanceItemCommentDeleteFailed = status.Error(codes.Internal, "删除绩效评论失败")

	// 评分
	ErrPerformanceAdditionScoreFailed           = status.Error(codes.Internal, "给其他同事评分失败")
	ErrPerformanceAdditionScoreAlreadyEvaluated = status.Error(codes.Internal, "给其他同事已经评价过了")

	// 绩效排名 - 排序列
	PerformanceSortFields = map[string]struct{}{"total_score": {}, "total_bonus": {}}
	SortOrders            = map[string]struct{}{"asc": {}, "desc": {}}
)

// 生成绩效lark消息通知的标题
func genPerformanceLarkMsgTitle(year, monthNum uint64) string {
	monthStr := fmt.Sprintf("%d~%d月", 2*monthNum-1, 2*monthNum)
	if monthNum <= 0 || monthNum > 6 {
		monthStr = ""
	}
	return fmt.Sprintf("%d年%s绩效考核", year, monthStr)
}

func extractEmailUsername(email string) string {
	if email == "" {
		return email
	}

	ret := strings.Split(email, "@")
	return ret[0]
}

// 校验自评分(0,150]
func isSelfScoreValid(s float64) error {
	if s <= minPerformanceScore || s > maxPerformanceScore {
		return status.Errorf(codes.InvalidArgument, "无效的自评分")
	}
	return nil
}

// 校验上级评分[0,150]
func isLeaderScoreValid(s float64) error {
	if s < minPerformanceScore || s > maxPerformanceScore {
		return status.Errorf(codes.InvalidArgument, "无效的上级评分")
	}
	return nil
}

// 校验绩效项权重(0,75]
func isItemWeightValid(w float64) error {
	if w < 0 || w > staffWorkWeightTotal {
		return status.Errorf(codes.InvalidArgument, "无效的绩效项目权重")
	}
	return nil
}

// get follower users
func getFollowerEmailList(l []*usercenter.PublicLbkUserOutside) []string {
	emailList := make([]string, len(l))
	for i, v := range l {
		emailList[i] = v.Email
	}
	return emailList
}

func getPerformanceLevelByScore(s float64) (string, float64) {
	switch {
	case s >= maxPerformanceScore:
		return PerformanceLevelA, PerformanceLevelAFactor
	case s >= 120:
		return PerformanceLevelBPlus, PerformanceLevelBPlusFactor
	case s >= 100:
		return PerformanceLevelB, PerformanceLevelBFactor
	case s >= 80:
		return PerformanceLevelC, PerformanceLevelCFactor
	case s >= 60:
		return PerformanceLevelD, PerformanceLevelDFactor
	default:
		return PerformanceLevelE, PerformanceLevelEFactor
	}
}
