package performancerpclogic

import (
	"context"
	"errors"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type CreatePerformanceItemCommentLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewCreatePerformanceItemCommentLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreatePerformanceItemCommentLogic {
	return &CreatePerformanceItemCommentLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 新建评论
func (l *CreatePerformanceItemCommentLogic) CreatePerformanceItemComment(in *oa.CreatePerformanceItemCommentReq) (*oa.CreatePerformanceItemCommentResp, error) {
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		logx.Errorw("failed to query x_performance on CreatePerformanceItemComment", logx.Field("error", err), logx.Field("performance_id", in.GetPerformanceId()))
		return nil, ErrPerformanceQueryFailed
	}
	// 绩效结束时不允许修改
	if p.Status == uint64(oa.PerformanceStatus_DONE) {
		return nil, ErrPerformanceCanNotBeModified
	}

	// 获取用户详情
	userDetail, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on CreatePerformanceItemComment", logx.Field("error", err), logx.Field("email", p.UserEmail))
		return nil, common.ErrUsercenterUnavailable
	}

	if p.UserEmail != in.GetUserEmail() && userDetail.LeaderEmail != in.GetUserEmail() {
		return nil, ErrPermissionDenied
	}

	switch in.GetCommentType() {
	case oa.PerformanceCommentType_WORK_ITEM:
		_, err = l.svcCtx.PerformanceWorkModel.XFindOne(l.ctx, in.GetItemId())
		if err != nil {
			if errors.Is(err, performance.ErrNotFound) {
				return nil, ErrPerformanceWorkItemNotFound
			}
			logx.Errorw("failed to query x_performance_work on CreatePerformanceItemComment", logx.Field("error", err))
			return nil, ErrPerformanceWorkItemQueryFailed
		}
	case oa.PerformanceCommentType_ADDITION_ITEM:
		_, err = l.svcCtx.PerformanceAddtionModel.XFindOne(l.ctx, in.GetItemId())
		if err != nil {
			if errors.Is(err, performance.ErrNotFound) {
				return nil, ErrPerformanceAdditionItemNotFound
			}
			return nil, ErrPerformanceAdditionItemQueryFailed
		}
	default:
		return nil, status.Error(codes.InvalidArgument, "无效的item_type")
	}

	// 创建评论
	comment := &performance.XPerformanceItemComment{
		PerformanceId: in.GetPerformanceId(),
		ItemId:        in.GetItemId(),
		CommentType:   uint64(in.GetCommentType()),
		UserEmail:     in.GetUserEmail(),
		Comment:       in.GetComment(),
	}
	ret, err := l.svcCtx.PerformanceItemCommentModel.XInsert(l.ctx, comment)
	if err != nil {
		logx.Errorw("failed to create x_performnace_item_comment on CreatePerformanceItemComment", logx.Field("error", err))
		return nil, ErrPerformanceItemCommentCreateFailed
	}
	commentId, _ := ret.LastInsertId()

	return &oa.CreatePerformanceItemCommentResp{CommentId: uint64(commentId)}, nil
}
