package performancerpclogic

import (
	"context"
	"errors"
	"fmt"
	"oa-server/common/globalkey"
	"time"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

const (
	joinTimeDay = 15 //入职时间15号
)

type CreatePerformanceLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewCreatePerformanceLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreatePerformanceLogic {
	return &CreatePerformanceLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 创建绩效
func (l *CreatePerformanceLogic) CreatePerformance(in *oa.CreatePerformanceReq) (*oa.CreatePerformanceResp, error) {
	if in.GetUserEmail() == "" || in.GetYear() == 0 || in.GetMonthNum() == 0 || in.GetMonthNum() > 6 {
		return nil, status.Error(codes.InvalidArgument, "无效的参数")
	}

	p, err := l.svcCtx.PerformanceModel.GetOneByUserAndTime(l.ctx, in.GetUserEmail(), in.GetYear(), in.GetMonthNum())
	if err != nil && !errors.Is(err, performance.ErrNotFound) {
		logx.Errorw("failed to query table x_performance on CreatePerformance", logx.Field("error", err))
		return nil, ErrPerformanceQueryFailed
	}

	// todo 并发
	resp := new(oa.CreatePerformanceResp)
	if p != nil {
		// 已经存在绩效则返回，不创建
		resp.Id = p.Id
		return resp, nil
	}

	// 获取用户详情
	userResp, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: in.GetUserEmail()})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on CreatePerformance", logx.Field("error", err), logx.Field("email", in.GetUserEmail()))
		return nil, common.ErrUsercenterUnavailable
	}
	t, _ := getDoubleMonthTimeRange(int(in.GetYear()), int(in.GetMonthNum()))
	// 时间戳转换为时间对象
	joinTime := time.Unix(userResp.JoinTime, 0).Local()
	fmt.Println(userResp.JoinTime, time.Unix(userResp.JoinTime, 0), time.Unix(userResp.JoinTime, 0).Local(), t.Second.Start.AddDate(0, 0, 14), joinTime.After(t.Second.Start.AddDate(0, 0, 14)))
	if joinTime.After(t.Second.Start.AddDate(0, 0, 14)) {
		return nil, ErrPerformanceNewEmployeeCreateFailed
	}
	// 获取用户绩效时间系数
	timeFactor, _ := getUserPerformanceTimeFactor(userResp.JoinTime, int64(in.GetYear()), int64(in.GetMonthNum()))

	// 创建绩效
	p = &performance.XPerformance{
		UserEmail:           in.GetUserEmail(),
		LeaderEmail:         userResp.LeaderEmail,
		Year:                in.GetYear(),
		MonthNum:            in.GetMonthNum(),
		WorkItemWeight:      staffWorkWeightTotal,
		AdditionItemWeight:  staffAdditionWeightTotal,
		Status:              uint64(oa.PerformanceStatus_WORKLOAD_FILLING),
		TimeFactor:          timeFactor,
		IsExcellentEmployee: uint64(oa.IsExcellentEmployee_IS_EXCELLENT_EMPLOYEE_NO),
	}
	if len(userResp.OwnerDepartmentList) > 0 {
		p.WorkItemWeight = managerWorkWeightTotal
		p.AdditionItemWeight = managerAdditionWeightTotal
	}

	l.svcCtx.PerformanceModel.TransCtx(l.ctx, func(ctx context.Context, s sqlx.Session) error {
		// 创建performance
		ret, err := l.svcCtx.PerformanceModel.TransInsertCtx(ctx, s, p)
		if err != nil {
			logx.Errorw("failed to create performance on CreatePreformance", logx.Field("error", err))
			return ErrPerformanceCreateFailed
		}
		performanID, err := ret.LastInsertId()
		if err != nil {
			logx.Errorw("failed to get performance id after db insert on CreatePerformance", logx.Field("error", err))
			return ErrPerformanceCreateFailed
		}
		p.Id = uint64(performanID)

		additionItems := genStaffPerfromanceAdditionItems(uint64(performanID))
		if len(userResp.OwnerDepartmentList) > 0 {
			// OwnerDepartmentList不为空，说明是领导
			additionItems = genLeaderPerfromanceAdditionItems(uint64(performanID))
		}

		// 创建附加项
		for _, v := range additionItems {
			_, err = l.svcCtx.PerformanceAddtionModel.TransInsertCtx(l.ctx, s, &v)
			if err != nil {
				logx.Errorw("Failed to create performance_addition on CreatePerformance", logx.Field("error", err))
				return ErrPerformanceAdditionItemCreateFailed
			}
		}
		return nil
	})

	return &oa.CreatePerformanceResp{Id: p.Id}, nil
}

// 生成普通员工绩效附加项
func genStaffPerfromanceAdditionItems(performanceId uint64) []performance.XPerformanceAddition {
	return []performance.XPerformanceAddition{
		{
			PerformanceId: performanceId,
			ItemType:      uint64(oa.AdditionItemType_BEHAVIOR_AND_ATTITUDE),
			ItemName:      "制度/执行力/工作时长",
			ItemWeight:    10,
		},
		{
			PerformanceId: performanceId,
			ItemType:      uint64(oa.AdditionItemType_ADDITIONAL_RULES),
			ItemName:      "生产bug或事故数量",
			ItemWeight:    10,
		},
		{
			PerformanceId: performanceId,
			ItemType:      uint64(oa.AdditionItemType_ADDITIONAL_RULES),
			ItemName:      additionItemCoworker,
			ItemWeight:    5,
		},
	}
}

// 生成管理人员绩效附加项
func genLeaderPerfromanceAdditionItems(performanceId uint64) []performance.XPerformanceAddition {
	return []performance.XPerformanceAddition{
		{
			PerformanceId: performanceId,
			ItemType:      uint64(oa.AdditionItemType_BEHAVIOR_AND_ATTITUDE),
			ItemName:      "制度/执行力/工作时长",
			ItemWeight:    10,
		},
		{
			PerformanceId: performanceId,
			ItemType:      uint64(oa.AdditionItemType_MANAGEMENT),
			ItemName:      "团队管理/成长/执行力",
			ItemWeight:    10,
		},
		{
			PerformanceId: performanceId,
			ItemType:      uint64(oa.AdditionItemType_ADDITIONAL_RULES),
			ItemName:      "生产bug或事故数量",
			ItemWeight:    5,
		},
		{
			PerformanceId: performanceId,
			ItemType:      uint64(oa.AdditionItemType_ADDITIONAL_RULES),
			ItemName:      additionItemCoworker,
			ItemWeight:    5,
		},
	}
}

type timeRange struct {
	Start time.Time
	End   time.Time
}
type doubleMonth struct {
	First  timeRange
	Second timeRange
}

func getDoubleMonthTimeRange(year, monthNum int) (doubleMonth, error) {
	// >6, <=0就返回全年的时间
	if monthNum > 6 || monthNum <= 0 {
		return doubleMonth{}, errors.New("invalid monthNum")
	}
	//返回指定双月的时间范围
	return doubleMonth{
		First:  getYearMonthTimeRange(year, 2*monthNum-1),
		Second: getYearMonthTimeRange(year, 2*monthNum),
	}, nil
}

// 获取指定年月的起止时间
func getYearMonthTimeRange(year, month int) timeRange {
	firstDay := time.Date(year, time.Month(month), 1, 0, 0, 0, 0, time.Local)
	lastDay := firstDay.AddDate(0, 1, 0).Add(-time.Nanosecond)
	return timeRange{Start: firstDay, End: lastDay}
}

func getUserPerformanceTimeFactor(joinTime, year, monthNum int64) (float64, error) {
	jt := time.Unix(joinTime, 0)
	doubleMonth, err := getDoubleMonthTimeRange(int(year), int(monthNum))
	if err != nil {
		return 0, err
	}
	switch {
	case jt.Before(doubleMonth.First.Start):
		return 2, nil
	case jt.Before(doubleMonth.First.Start.AddDate(0, 0, joinTimeDay-1)):
		return 1.5, nil
	case jt.Before(doubleMonth.Second.Start):
		return 1, nil
	case jt.Before(doubleMonth.Second.Start.AddDate(0, 0, joinTimeDay-1)):
		return 0.5, nil
	}
	return 0, nil
}

// 批量创建绩效
func (l *CreatePerformanceLogic) BatchCreatePerformance(year, month_num uint64) error {
	users, err := l.svcCtx.LbkUcClient.GetDepartmentDirectlyUser(context.Background(), usercenter.GetDepartmentDirectlyUserReq{
		AppKey:        l.svcCtx.Config.LbkUsercenter.AppName,
		AppToken:      l.svcCtx.Config.LbkUsercenter.AppToken,
		DepartmentKey: globalkey.RDDepartmentKey,
	})
	fmt.Println(l.svcCtx.Config.LbkUsercenter.AppName, l.svcCtx.Config.LbkUsercenter.AppToken)
	//users, err := l.svcCtx.LbkUcClient.GetDepartmentUser(context.Background(), usercenter.GetDepartmentUserReq{
	//	DepartmentKeyList: []string{
	//		globalkey.RDDepartmentKey,
	//	},
	//})
	if err != nil {
		fmt.Println("获取部门用户出错", err)
		return err
	}

	if month_num == 0 {
		logx.Errorw("invalid monthNum", logx.Field("monthNum", month_num))
		fmt.Println("当前月份不是双月，不创建绩效", month_num)
		return errors.New("invalid monthNum,当前月份不是双月，不创建绩效")
	}
	for k, user := range users {
		if user.Email == globalkey.RockyEmail {
			fmt.Println("跳过Rocky", user.Email, k)
			continue
		}
		fmt.Println("开始创建绩效", user.Email, year, month_num, time.Now())
		p, err := l.svcCtx.PerformanceModel.GetOneByUserAndTime(context.Background(), user.Email, year, month_num)
		if err != nil && !errors.Is(err, performance.ErrNotFound) {
			logx.Error(err)
			continue
		}
		if p != nil {
			fmt.Println(user.Email, year, month_num, "绩效已经存在,绩效ID：", p.Id, time.Now())
			continue
		}
		//按照部门循环每个部门下面的员工，创建绩效
		resp, err := l.CreatePerformance(&oa.CreatePerformanceReq{
			UserEmail: user.Email,
			Year:      year,
			MonthNum:  month_num,
		})
		if err != nil {
			fmt.Println("创建绩效出错，继续跑下一条", err)
			continue
		}
		// 发送lark通知
		//common.AsyncSendLarkMsg(l.svcCtx.Config.Env, func() {
		//	msgData := map[string]any{
		//		"title":      "系统已为您自动创建" + genPerformanceLarkMsgTitle(uint64(year), mapYearMonth[month]),
		//		"content":    fmt.Sprintf("%s 您好，系统已为您自动创建"+genPerformanceLarkMsgTitle(uint64(year), mapYearMonth[month])+"，请及时填写", extractEmailUsername(user.Email)),
		//		"target_url": l.svcCtx.Config.Lark.MyPerformanceUrl,
		//	}
		//	msg, err := l.svcCtx.LarkService.BuildCardContent(l.svcCtx.Config.Lark.PerformanceTmpId, msgData)
		//	if err != nil {
		//		l.Logger.Errorw("failed to build lark card content", logx.Field("error", err))
		//		return
		//	}
		//	if err = l.svcCtx.LarkService.SendCardMessage(msgcenter.LarkReceiveIdTypeEmail, user.Email, msgcenter.LarkMsgTypeCard, msg); err != nil {
		//		l.Logger.Errorw("failed to send user lark msg", logx.Field("error", err))
		//		return
		//	}
		//})
		fmt.Println(user.Email, year, month_num, "自动创建绩效成功", k, resp.Id, time.Now())
	}
	return nil
}
