package performancerpclogic

import (
	"context"
	"errors"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/shopspring/decimal"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type CreatePerformanceWorkItemLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewCreatePerformanceWorkItemLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreatePerformanceWorkItemLogic {
	return &CreatePerformanceWorkItemLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 创建绩效工作部分
func (l *CreatePerformanceWorkItemLogic) CreatePerformanceWorkItem(in *oa.CreatePerformanceWorkItemReq) (*oa.CreatePerformanceWorkItemResp, error) {
	// 参数校验
	if err := isItemWeightValid(in.GetItemWeight()); err != nil {
		return nil, err
	}
	if err := isSelfScoreValid(in.GetSelfScore()); err != nil {
		return nil, err
	}

	// 获取performance
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		logx.Errorw("failed to query table x_performance on CreatePerformanceWorkItem", logx.Field("id", in.GetPerformanceId()), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}

	// 鉴权
	if p.UserEmail != in.GetUserEmail() {
		return nil, ErrPermissionDenied
	}

	if p.Status == uint64(oa.PerformanceStatus_NEED_LEADER_ACK) || p.Status == uint64(oa.PerformanceStatus_DONE) {
		// 待上级确认、结束 这两个状态是不允许修改的  4,5
		return nil, ErrPerformanceCanNotBeModified
	}

	// 获取用户信息
	userDetail, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on CreatePerformanceWorkItem", logx.Field("email", p.UserEmail), logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	// 计算工作部分权重和是否超过指定范围
	// 管理人员 70%  普通员工 75%
	workItems, err := l.svcCtx.PerformanceWorkModel.GetWorkItemsByPerformanceId(l.ctx, p.Id)
	if err != nil {
		logx.Errorw("failed to query table x_performance_work on CreatePerformanceWorkItem", logx.Field("error", err), logx.Field("performance_id", p.Id))
		return nil, ErrPerformanceWorkItemQueryFailed
	}
	weightTotal := sumWorkItemWeight(workItems)
	dWeight := decimal.NewFromFloat(in.GetItemWeight())
	if len(userDetail.OwnerDepartmentList) > 0 {
		// 领导工作部分权重和为70%
		if weightTotal.Add(dWeight).GreaterThan(decimal.NewFromFloat(managerWorkWeightTotal)) {
			return nil, status.Error(codes.InvalidArgument, "工作部分权重和最高为70%")
		}
	} else {
		// 普通员工工作部分权重和为75%
		if weightTotal.Add(dWeight).GreaterThan(decimal.NewFromFloat(staffWorkWeightTotal)) {
			return nil, status.Error(codes.InvalidArgument, "工作部分权重和最高为75%")
		}
	}

	// 创建
	data := performance.XPerformanceWork{
		PerformanceId:  in.GetPerformanceId(),
		ItemName:       in.GetItemName(),
		ItemWeight:     in.GetItemWeight(),
		ItemDesc:       in.GetItemDesc(),
		ItemDeficiency: in.GetItemDeficency(),
		SelfScore:      in.GetSelfScore(),
	}

	ret, err := l.svcCtx.PerformanceWorkModel.XInsert(l.ctx, &data)
	if err != nil {
		logx.Errorw("failed to create performance work item", logx.Field("error", err))
		return nil, ErrPerformanceWorkItemCreateFailed
	}
	itemID, _ := ret.LastInsertId()

	return &oa.CreatePerformanceWorkItemResp{ItemId: uint64(itemID)}, nil
}

func sumWorkItemWeight(items performance.PerformanceWorkItemList) decimal.Decimal {
	weightTotal := decimal.NewFromFloat(0)
	for _, v := range items {
		weightTotal = weightTotal.Add(decimal.NewFromFloat(v.ItemWeight))
	}

	return weightTotal
}
