package performancerpclogic

import (
	"context"
	"errors"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type DeletePerformanceWorkItemLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewDeletePerformanceWorkItemLogic(ctx context.Context, svcCtx *svc.ServiceContext) *DeletePerformanceWorkItemLogic {
	return &DeletePerformanceWorkItemLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 删除绩效工作部分
func (l *DeletePerformanceWorkItemLogic) DeletePerformanceWorkItem(in *oa.DeletePerformanceWorkItemReq) (*oa.DeletePerformanceWorkItemResp, error) {
	workItem, err := l.svcCtx.PerformanceWorkModel.XFindOne(l.ctx, in.GetItemId())
	if err != nil {
		l.Logger.Errorw("failed to query table x_performance_work on DeletePerformanceWorkItem", logx.Field("id", in.GetItemId()), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceWorkItemNotFound
		}
		return nil, ErrPerformanceWorkItemQueryFailed
	}

	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, workItem.PerformanceId)
	if err != nil {
		l.Logger.Errorw("failed to query table x_performance on DeletePerformanceWorkItem", logx.Field("id", workItem.PerformanceId), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}
	if p.UserEmail != in.GetUserEmail() {
		return nil, ErrPermissionDenied
	}

	if p.Status == uint64(oa.PerformanceStatus_NEED_LEADER_ACK) || p.Status == uint64(oa.PerformanceStatus_DONE) {
		// 待上级确认、结束 这两个状态是不允许修改的  4,5
		return nil, ErrPerformanceCanNotBeModified
	}

	// 查询该工作项的评论
	pcList, err := l.svcCtx.PerformanceItemCommentModel.FindByPerformanceItem(l.ctx, p.Id, workItem.Id, uint32(oa.PerformanceCommentType_WORK_ITEM))
	if err != nil {
		l.Logger.Errorw("failed to query x_performance_item_comment on DeletePerformanceWorkItem", logx.Field("error", err))
		return nil, ErrPerformanceItemCommentQueryFailed
	}

	// 开启事物删除工作项, 评论
	if err = l.svcCtx.PerformanceWorkModel.TransCtx(l.ctx, func(ctx context.Context, s sqlx.Session) error {
		if err := l.svcCtx.PerformanceWorkModel.TransDeleteCtx(ctx, s, workItem.Id); err != nil {
			l.Logger.Errorw("failed to delete performance work item on DeletePerformanceWorkItem", logx.Field("id", workItem.Id), logx.Field("error", err))
			return ErrPerformanceWorkItemDeleteFailed
		}
		if len(pcList) > 0 {
			if err := l.svcCtx.PerformanceItemCommentModel.TransMultiDeleteCtx(ctx, s, pcList.Ids()); err != nil {
				l.Logger.Errorw("failed to delete x_performance_item_comment on DeletePerformanceWorkItem", logx.Field("error", err))
				return ErrPerformanceItemCommentDeleteFailed
			}
		}
		return nil
	}); err != nil {
		return nil, err
	}

	return &oa.DeletePerformanceWorkItemResp{}, nil
}
