package performancerpclogic

import (
	"context"
	"errors"
	"fmt"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/msgcenter"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/shopspring/decimal"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type FinishLeaderEvaluationLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewFinishLeaderEvaluationLogic(ctx context.Context, svcCtx *svc.ServiceContext) *FinishLeaderEvaluationLogic {
	return &FinishLeaderEvaluationLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 上级完成评价
func (l *FinishLeaderEvaluationLogic) FinishLeaderEvaluation(in *oa.FinishLeaderEvaluationReq) (*oa.FinishLeaderEvaluationResp, error) {
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		l.Logger.Errorw("faield to query table x_performance on FinishLeaderEvaluation", logx.Field("id", in.GetPerformanceId()), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}

	// 获取用户信息
	userDetail, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on FinishLeaderEvaluation", logx.Field("email", p.UserEmail), logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	if userDetail.LeaderEmail != in.GetUserEmail() {
		return nil, ErrPermissionDenied
	}

	// 检查当前绩效状态
	if p.Status == uint64(oa.PerformanceStatus_DONE) {
		return nil, ErrPerformanceCanNotBeModified
	}

	// 检查上级评价是否填写
	if p.LeaderWords == "" {
		return nil, status.Error(codes.FailedPrecondition, "上级评价未填写")
	}

	// 获取workItem,additionItem
	workItems, err := l.svcCtx.PerformanceWorkModel.GetWorkItemsByPerformanceId(l.ctx, p.Id)
	if err != nil {
		l.Logger.Errorw("failed to query x_performance_work on FinishLeaderEvaluation", logx.Field("performance_id", p.Id), logx.Field("error", err))
		return nil, ErrPerformanceWorkItemQueryFailed
	}
	additionItems, err := l.svcCtx.PerformanceAddtionModel.GetAdditionItemsByPerformandId(l.ctx, p.Id)
	if err != nil {
		l.Logger.Errorw("failed to query x_performance_addition on FinishLeaderEvaluation", logx.Field("performance_id", p.Id), logx.Field("error", err))
		return nil, ErrPerformanceAdditionItemQueryFailed
	}

	// 计算绩效总分, 并更新绩效状态、总分
	p.TotalScore = caculatePerformanceScore(workItems, additionItems)
	level, levelFactor := getPerformanceLevelByScore(p.TotalScore)
	p.Level = level
	p.LevelFactor = levelFactor
	if p.Status == uint64(oa.PerformanceStatus_LEADER_EVALUATION) {
		p.Status = uint64(oa.PerformanceStatus_NEED_STAFF_ACK)
	}
	if err = l.svcCtx.PerformanceModel.XUpdate(l.ctx, p); err != nil {
		l.Logger.Errorw("failed to udpate performance totalScore and status on FinishLeaderEvaluation", logx.Field("error", err), logx.Field("id", p.Id))
		return nil, ErrPerformanceUpdateFailed
	}

	// 发送lark通知
	common.AsyncSendLarkMsg(l.svcCtx.Config.Env, func() {
		msgData := map[string]any{
			"title":      genPerformanceLarkMsgTitle(p.Year, p.MonthNum),
			"content":    fmt.Sprintf("%s 已完成绩效评估，请及时确认", extractEmailUsername(userDetail.LeaderEmail)),
			"target_url": l.svcCtx.Config.Lark.MyPerformanceUrl,
		}
		msg, err := l.svcCtx.LarkService.BuildCardContent(l.svcCtx.Config.Lark.PerformanceTmpId, msgData)
		if err != nil {
			l.Logger.Errorw("failed to build lark card content", logx.Field("error", err))
			return
		}
		if err = l.svcCtx.LarkService.SendCardMessage(msgcenter.LarkReceiveIdTypeEmail, p.UserEmail, msgcenter.LarkMsgTypeCard, msg); err != nil {
			l.Logger.Errorw("failed to send user lark msg", logx.Field("error", err))
			return
		}
	})

	return &oa.FinishLeaderEvaluationResp{}, nil
}

// 计算绩效总分
func caculatePerformanceScore(workItems performance.PerformanceWorkItemList, additionItems performance.PerformanceAdditionItemList) float64 {
	total := decimal.NewFromFloat(0)
	for _, v := range workItems {
		w := decimal.NewFromFloat(v.ItemWeight)
		s := decimal.NewFromFloat(v.LeaderScore)
		total = total.Add(w.Mul(s))
	}

	for _, v := range additionItems {
		w := decimal.NewFromFloat(v.ItemWeight)
		s := decimal.NewFromFloat(v.LeaderScore)
		total = total.Add(w.Mul(s))
	}

	// 总分再除100，并保留2位小数
	score, _ := total.DivRound(percent, 2).Float64()
	return score
}
