package performancerpclogic

import (
	"context"
	"errors"
	"fmt"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/msgcenter"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type FinishPerformanceWorkloadFillingLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewFinishPerformanceWorkloadFillingLogic(ctx context.Context, svcCtx *svc.ServiceContext) *FinishPerformanceWorkloadFillingLogic {
	return &FinishPerformanceWorkloadFillingLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 绩效填写、自评完成
func (l *FinishPerformanceWorkloadFillingLogic) FinishPerformanceWorkloadFilling(in *oa.FinishPerformanceWorkloadFillingReq) (*oa.FinishPerformanceWorkloadFillingResp, error) {
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		l.Logger.Errorw("failed to query x_performance", logx.Field("error", err), logx.Field("id", in.GetPerformanceId()))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}

	// 判断绩效归属
	if p.UserEmail != in.GetUserEmail() {
		return nil, ErrPermissionDenied
	}
	if p.Status == uint64(oa.PerformanceStatus_NEED_LEADER_ACK) || p.Status == uint64(oa.PerformanceStatus_DONE) {
		// 待上级确认、结束 这两个状态是不允许修改的  4,5
		return nil, ErrPerformanceCanNotBeModified
	}

	// 获取用户信息
	userDetail, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on FinishPerformanceWorkloadFilling", logx.Field("email", p.UserEmail), logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	// 检查workItem,additionItem
	workItems, err := l.svcCtx.PerformanceWorkModel.GetWorkItemsByPerformanceId(l.ctx, p.Id)
	if err != nil {
		l.Logger.Errorw("failed to query x_performance_work", logx.Field("performance_id", p.Id), logx.Field("error", err))
		return nil, ErrPerformanceWorkItemQueryFailed
	}
	additionItems, err := l.svcCtx.PerformanceAddtionModel.GetAdditionItemsByPerformandId(l.ctx, p.Id)
	if err != nil {
		l.Logger.Errorw("failed to query x_performance_addition", logx.Field("performance_id", p.Id), logx.Field("error", err))
		return nil, ErrPerformanceAdditionItemQueryFailed
	}
	if err = isWorkloadAndAdditionDone(workItems, additionItems); err != nil {
		l.Logger.Errorw("绩效工作部分、附加部分未填写完成", logx.Field("performance_id", p.Id), logx.Field("error", err))
		return nil, status.Error(codes.FailedPrecondition, "绩效工作部分、附加部分未填写完成")
	}

	if p.Status == uint64(oa.PerformanceStatus_WORKLOAD_FILLING) { //首次进来，被打回
		// 只有当绩效之前的状态是工作产出填写时，才更新绩效状态
		//p.Status = uint64(oa.PerformanceStatus_LEADER_EVALUATION)
		// 如果score！=0   如果没有环评，就走下面的逻辑

		// 添加环评表
		// 其他部门同事评价时，需要添加环评表
		for _, v := range additionItems {
			if v.ItemName == additionItemCoworker {
				if v.SelfScore != 0 { //不等于0说明是被领导打回的，已经评价过的，就不需要再评价了
					p.Status = uint64(oa.PerformanceStatus_LEADER_EVALUATION) //如果有评价过，跳过环评，直接眺到节点2
					if err = l.svcCtx.PerformanceModel.XUpdate(l.ctx, p); err != nil {
						l.Logger.Error("failed to update performance status", logx.Field("performance_id", p.Id), logx.Field("status", p.Status), logx.Field("error", err))
						return nil, ErrPerformanceUpdateFailed
					}
					//直接跳过环评，给上级领导发lark
					common.AsyncSendLarkMsg(l.svcCtx.Config.Env, func() {
						msgData := map[string]any{
							"title":      genPerformanceLarkMsgTitle(p.Year, p.MonthNum),
							"content":    fmt.Sprintf("%s 已完成绩效评估，请及时确认", extractEmailUsername(p.UserEmail)),
							"target_url": l.svcCtx.Config.Lark.ScorePerformanceUrl,
						}
						msg, err := l.svcCtx.LarkService.BuildCardContent(l.svcCtx.Config.Lark.PerformanceTmpId, msgData)
						if err != nil {
							l.Logger.Errorw("failed to build lark card content", logx.Field("error", err))
							return
						}
						if err = l.svcCtx.LarkService.SendCardMessage(msgcenter.LarkReceiveIdTypeEmail, userDetail.LeaderEmail, msgcenter.LarkMsgTypeCard, msg); err != nil {
							l.Logger.Errorw("failed to send user lark msg", logx.Field("error", err))
							return
						}
					})
				} else { //如果=0说明地第一次，正常评价
					performanceScore, _ := l.svcCtx.PerformanceAdditionScoreModel.XFindOne(l.ctx, v.Id)
					if performanceScore == nil {
						err = l.svcCtx.PerformanceModel.TransCtx(l.ctx, func(ctx context.Context, session sqlx.Session) error {
							//更新addition_score
							if _, err := l.svcCtx.PerformanceAdditionScoreModel.TransInsertCtx(ctx, session, &performance.XPerformanceAdditionScore{
								PerformanceId: v.PerformanceId,
								ItemId:        int64(v.Id),
								Score:         0,
								Year:          p.Year,
								MonthNum:      p.MonthNum,
								UserEmail:     p.UserEmail,
								Status:        1,
								ItemDesc:      "",
								CoworkerEmail: v.CoworkerEmail,
							}); err != nil {
								l.Logger.Errorw("failed to insert x_performance_addition_score itemDesc and Score", logx.Field("error", err), logx.Field("item_id", v.Id))
								return ErrPerformanceAdditionScoreFailed
							}
							//更新performance
							p.Status = uint64(oa.PerformanceStatus_COWORKER_EVALUATION)
							if err := l.svcCtx.PerformanceModel.TransUpdateCtx(ctx, session, p); err != nil {
								l.Logger.Errorw("failed to update x_performance status", logx.Field("error", err), logx.Field("performance_id", p.Id))
								return ErrPerformanceUpdateFailed
							}
							return nil
						})
						if err != nil {
							l.Logger.Errorw("failed to insert x_performance_addition_score", logx.Field("error", err), logx.Field("item_id", v.Id))
							return nil, ErrPerformanceAdditionScoreFailed
						}
						// 发送lark通知
						common.AsyncSendLarkMsg(l.svcCtx.Config.Env, func() {
							msgData := map[string]any{
								"title":      genPerformanceLarkMsgTitle(p.Year, p.MonthNum),
								"content":    fmt.Sprintf("%s 邀请您进行绩效评价，请及时处理", extractEmailUsername(p.UserEmail)),
								"target_url": l.svcCtx.Config.Lark.ScorePerformanceUrl,
							}
							msg, err := l.svcCtx.LarkService.BuildCardContent(l.svcCtx.Config.Lark.PerformanceTmpId, msgData)
							if err != nil {
								l.Logger.Errorw("failed to build lark card content", logx.Field("error", err))
								return
							}
							if err = l.svcCtx.LarkService.SendCardMessage(msgcenter.LarkReceiveIdTypeEmail, v.CoworkerEmail, msgcenter.LarkMsgTypeCard, msg); err != nil {
								l.Logger.Errorw("failed to send user lark msg", logx.Field("error", err))
								return
							}
						})
					} else {
						return nil, ErrPerformanceAdditionItemUpdateFailed
					}
				}
			}
		}
	}

	return &oa.FinishPerformanceWorkloadFillingResp{}, nil
}

func isWorkloadAndAdditionDone(workItems performance.PerformanceWorkItemList, additionItems performance.PerformanceAdditionItemList) error {
	if len(workItems) == 0 || len(additionItems) == 0 {
		return errors.New("绩效的工作项 和 附加项 为必填项")
	}
	for _, v := range workItems {
		if v.ItemWeight <= 0 {
			return fmt.Errorf("绩效工作部分权重无效, %v", v.ItemWeight)
		}
		if v.SelfScore <= 0 {
			return fmt.Errorf("绩效工作部分自评分无效, %v", v.SelfScore)
		}

		if v.ItemName == "" || v.ItemDesc == "" || v.ItemDeficiency == "" {
			return fmt.Errorf("绩效工作部分未填写完整")
		}
	}

	for _, v := range additionItems {
		if v.ItemWeight <= 0 {
			return fmt.Errorf("绩效附加部分权重无效, %v", v.ItemWeight)
		}
		if v.ItemName != additionItemCoworker { //如果不是其他部门评价，才会有这个判断
			if v.SelfScore <= 0 {
				return fmt.Errorf("绩效附加部分自评分无效, %v", v.SelfScore)
			}
			if v.ItemDesc == "" {
				return fmt.Errorf("绩效附加部分未填写完整")
			}
		} else {
			if v.CoworkerEmail == "" {
				return fmt.Errorf("被邀请人未填写, %v", v.CoworkerEmail)
			}
		}

	}
	return nil
}
