package performancerpclogic

import (
	"context"
	"errors"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	timestamppb "google.golang.org/protobuf/types/known/timestamppb"
)

type GetPerformanceDetailLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetPerformanceDetailLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetPerformanceDetailLogic {
	return &GetPerformanceDetailLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取绩效详情
func (l *GetPerformanceDetailLogic) GetPerformanceDetail(in *oa.GetPerformanceDetailReq) (*oa.GetPerformanceDetailResp, error) {
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		logx.Errorw("failed to query x_performance on GetPerformanceDetail", logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}

	// 查询绩效归属人信息
	user, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on GetPerformanceDetail", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	// 只有员工自己, 直接上级，Rocky可以查看绩效详情
	if !p.CanView(in.GetUserEmail(), user.LeaderEmail) {
		return nil, ErrPermissionDenied
	}
	scoreAndComment := true
	if p.UserEmail == in.GetUserEmail() && (p.Status == uint64(oa.PerformanceStatus_WORKLOAD_FILLING) || p.Status == uint64(oa.PerformanceStatus_LEADER_EVALUATION) || p.Status == uint64(oa.PerformanceStatus_COWORKER_EVALUATION)) {
		scoreAndComment = false // 员工自己查看自己的绩效时，且状态不是已提交，不显示上级评价
	}

	// performance work items
	workList, err := l.svcCtx.PerformanceWorkModel.GetWorkItemsByPerformanceId(l.ctx, p.Id)
	if err != nil {
		logx.Errorw("failed to query x_performance_work on GetPerformanceDetail", logx.Field("performance_id", p.Id), logx.Field("error", err))
		return nil, ErrPerformanceWorkItemQueryFailed
	}

	// performance addition items
	additionList, err := l.svcCtx.PerformanceAddtionModel.GetAdditionItemsByPerformandId(l.ctx, p.Id)
	if err != nil {
		logx.Errorw("failed to query x_performance_addition on GetPerformanceDetail", logx.Field("performance_id", p.Id), logx.Field("error", err))
		return nil, ErrPerformanceAdditionItemQueryFailed
	}

	// 绩效工作项、附加项评论
	pcList, err := l.svcCtx.PerformanceItemCommentModel.FindByPerformance(l.ctx, p.Id)
	if err != nil {
		logx.Errorw("failed to query x_performance_item_comment on GetPerformanceDetail", logx.Field("error", err))
		return nil, ErrPerformanceItemCommentQueryFailed
	}

	workItems := make([]*oa.PerformanceWorkItem, len(workList))
	for i, v := range workList {
		commentList := pcList.ByItemIdAndCommentType(v.Id, uint64(oa.PerformanceCommentType_WORK_ITEM))
		// 只显示最新的N条评论
		latestComments := getLatestComments(maxPerformanceItemCommentDisplayCnt, commentList)
		comments := make([]*oa.PerformanceItemComment, len(latestComments))
		for j, m := range latestComments {
			comments[j] = &oa.PerformanceItemComment{
				CommentId:   m.Id,
				ItemId:      m.ItemId,
				CommentType: oa.PerformanceCommentType(m.CommentType),
				UserEmail:   m.UserEmail,
				Comment:     m.Comment,
				CreatedAt:   timestamppb.New(m.CreatedAt),
				UpdatedAt:   timestamppb.New(m.UpdatedAt),
			}
		}
		leaderScore := v.LeaderScore
		if !scoreAndComment { //如果是员工自己查看自己的绩效时，且状态不是已提交，不显示上级评价
			leaderScore = 0
		}
		workItems[i] = &oa.PerformanceWorkItem{
			Id:            v.Id,
			ItemName:      v.ItemName,
			ItemWeight:    v.ItemWeight,
			ItemDesc:      v.ItemDesc,
			ItemDeficency: v.ItemDeficiency,
			SelfScore:     v.SelfScore,
			LeaderScore:   leaderScore,
			Comments:      comments,
		}
	}

	additionItems := make([]*oa.PerformanceAdditionItem, len(additionList))
	for i, v := range additionList {
		commentList := pcList.ByItemIdAndCommentType(v.Id, uint64(oa.PerformanceCommentType_ADDITION_ITEM))
		// 只显示最新的N条评论
		latestComments := getLatestComments(maxPerformanceItemCommentDisplayCnt, commentList)
		comments := make([]*oa.PerformanceItemComment, len(latestComments))
		for j, m := range latestComments {
			comments[j] = &oa.PerformanceItemComment{
				CommentId:   m.Id,
				ItemId:      m.ItemId,
				CommentType: oa.PerformanceCommentType(m.CommentType),
				UserEmail:   m.UserEmail,
				Comment:     m.Comment,
				CreatedAt:   timestamppb.New(m.CreatedAt),
				UpdatedAt:   timestamppb.New(m.UpdatedAt),
			}
		}
		additionLeaderScore := v.LeaderScore
		if !scoreAndComment { //如果是员工自己查看自己的绩效时，且状态不是已提交，不显示上级评价
			additionLeaderScore = 0
		}
		additionItems[i] = &oa.PerformanceAdditionItem{
			Id:            v.Id,
			ItemName:      v.ItemName,
			ItemType:      oa.AdditionItemType(v.ItemType),
			ItemWeight:    v.ItemWeight,
			ItemDesc:      v.ItemDesc,
			SelfScore:     v.SelfScore,
			LeaderScore:   additionLeaderScore,
			CoworkerEmail: v.CoworkerEmail,
			Comments:      comments,
		}
	}
	leaderWords := p.LeaderWords
	totalScore := p.TotalScore
	if !scoreAndComment { //如果是员工自己查看自己的绩效时，且状态不是已提交，不显示上级评价
		leaderWords = ""
		totalScore = 0
	}
	status := p.Status
	if p.Status == uint64(oa.PerformanceStatus_NEED_CONFIRM) || p.Status == uint64(oa.PerformanceStatus_SUBMITTED) { //首次进来，被打回
		status = uint64(oa.PerformanceStatus_DONE) //50以后代表已经结束了，就显示已完成
	}
	resp := &oa.GetPerformanceDetailResp{
		Performance: &oa.Performance{
			Id:                 p.Id,
			UserEmail:          p.UserEmail,
			Year:               p.Year,
			MonthNum:           p.MonthNum,
			WorkItemWeight:     p.WorkItemWeight,
			AdditionItemWeight: p.AdditionItemWeight,
			TotalScore:         totalScore,
			Level:              p.Level,
			LeaderWords:        leaderWords,
			Status:             oa.PerformanceStatus(status),
			CreatedAt:          timestamppb.New(p.CreatedAt),
			UpdatedAt:          timestamppb.New(p.UpdatedAt),
		},
		WorkItems:     workItems,
		AdditionItems: additionItems,
	}

	// 填充部门信息，取departmentList中第一个
	if len(user.DepartmentList) > 0 {
		resp.Performance.Department = &oa.DepartmentInfo{
			Key:    user.DepartmentList[0].DepartmentKey,
			Name:   user.DepartmentList[0].DepartmentName,
			Leader: user.LeaderEmail,
		}
	}

	return resp, nil
}

// 获取最新的N条评论
func getLatestComments(n int, comments performance.PerformanceItemCommentList) performance.PerformanceItemCommentList {
	if len(comments) <= n {
		return comments
	}
	return comments[len(comments)-n:]
}
