package performancerpclogic

import (
	"context"
	"errors"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/protobuf/types/known/timestamppb"
)

type GetPerformanceItemCommentListLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetPerformanceItemCommentListLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetPerformanceItemCommentListLogic {
	return &GetPerformanceItemCommentListLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 项目评论列表
func (l *GetPerformanceItemCommentListLogic) GetPerformanceItemCommentList(in *oa.GetPerformanceItemCommentListReq) (*oa.GetPerformanceItemCommentListResp, error) {
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		l.Logger.Errorw("failed to query x_performance on GetPerformanceItemCommentList", logx.Field("error", err))
	}

	// 查询绩效归属人信息
	user, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on GetPerformanceItemCommentList", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	// 员工自己，直接上级，Rocky可查看
	if !p.CanView(in.GetUserEmail(), user.LeaderEmail) {
		return nil, ErrPermissionDenied
	}

	list, err := l.svcCtx.PerformanceItemCommentModel.PageByPerformanceItem(l.ctx, performance.PageByPerformanceItemReq{
		PerformanceId: in.GetPerformanceId(),
		ItemId:        in.GetItemId(),
		CommentType:   uint32(in.GetCommentType()),
		PageNum:       in.GetPageNum(),
		PageSize:      in.GetPageSize(),
		OrderBy:       in.GetOrderBy(),
	})
	if err != nil {
		l.Logger.Errorw("failed to query x_performance_item_comment on GetPerformanceItemCommentList", logx.Field("error", err))
		return nil, ErrPerformanceItemCommentQueryFailed
	}

	total := len(list)
	comments := make([]*oa.PerformanceItemComment, total)
	for i, v := range list {
		comments[i] = &oa.PerformanceItemComment{
			CommentId:   v.Id,
			ItemId:      v.ItemId,
			CommentType: oa.PerformanceCommentType(v.CommentType),
			UserEmail:   v.UserEmail,
			Comment:     v.Comment,
			CreatedAt:   timestamppb.New(v.CreatedAt),
			UpdatedAt:   timestamppb.New(v.UpdatedAt),
		}
	}
	return &oa.GetPerformanceItemCommentListResp{Total: uint64(total), List: comments}, nil
}
