package performancerpclogic

import (
	"context"
	"errors"
	"fmt"
	"slices"
	"sort"
	"strconv"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/globalkey"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/timestamppb"
)

type GetPerformanceRankingLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetPerformanceRankingLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetPerformanceRankingLogic {
	return &GetPerformanceRankingLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取绩效排名
func (l *GetPerformanceRankingLogic) GetPerformanceRanking(in *oa.GetPerformanceRankingReq) (*oa.GetPerformanceRankingResp, error) {
	if _, ok := PerformanceSortFields[in.GetSortField()]; !ok {
		return nil, status.Error(codes.InvalidArgument, "无效的排序字段")
	}
	if _, ok := SortOrders[in.GetSortOrder()]; !ok {
		return nil, status.Error(codes.InvalidArgument, "无效的排序方式")
	}

	countData := new(performance.RankingCountData)
	var pList performance.PerformanceList
	var workItemList performance.PerformanceWorkItemList
	var err error
	var followers []*usercenter.PublicLbkUserOutside

	switch in.GetUserEmail() {
	case globalkey.RockyEmail:
		departmentKey := in.GetDepartmentKey()
		if departmentKey == "" {
			departmentKey = globalkey.RDDepartmentKey
		}
		getDepartmentDirectlyUserReq := usercenter.GetDepartmentDirectlyUserReq{
			AppKey:        l.svcCtx.Config.LbkUsercenter.AppName,
			AppToken:      l.svcCtx.Config.LbkUsercenter.AppToken,
			DepartmentKey: departmentKey,
		}

		followers, err = l.svcCtx.LbkUcClient.GetDepartmentDirectlyUser(l.ctx, getDepartmentDirectlyUserReq)
		if err != nil {
			logx.Errorw("failed to call usercenter.GetDepartmentDirectlyUser on GetPerformanceRanking", logx.Field("error", err))
			return nil, common.ErrUsercenterUnavailable
		}

	default:
		// 查询其下属，如果没有下属则其不是领导
		followers, err = l.svcCtx.LbkUcClient.GetLbkUserFollower(l.ctx, usercenter.LbkUserFllowerReq{Email: in.GetUserEmail()})
		if err != nil {
			l.Logger.Errorw("failed to query usercenter.GetLbkUserFollower on GetPerformanceRanking", logx.Field("error", err))
			return nil, common.ErrUsercenterUnavailable
		}
	}
	if len(followers) == 0 {
		l.Logger.Infow("user followers is empty on GetPerformanceRanking")
		return &oa.GetPerformanceRankingResp{}, nil
	}
	emailList := getFollowerEmailList(followers)

	//if in.GetUserEmail() != globalkey.RockyEmail {
	//	emailList = append(emailList, in.GetUserEmail())
	//}

	// 查询指定人员
	if in.GetSearchUserEmail() != "" {
		// 只能查看自己的直接下属，除了rocky
		if !slices.Contains(emailList, in.GetSearchUserEmail()) && in.GetUserEmail() != globalkey.RockyEmail {
			l.Logger.Infow("user followers not include SearchUserEmail", logx.Field("search_user_email", in.GetSearchUserEmail()))
			return nil, ErrPermissionDenied
		}
		return l.getPerformanceRankingForOneUser(in)
	}
	// 查询直接下属
	// 查询总量
	countData, err = l.svcCtx.PerformanceModel.CountByUserAndTime(l.ctx, emailList, in.GetYear(), in.GetMonthNum(), in.GetIsExcellentEmployee(), in.GetStatus())
	if err != nil {
		l.Logger.Errorw("failed to query x_performance on GetPerformanceRanking.CountByUserAndTime", logx.Field("error", err))
		return nil, ErrPerformanceQueryFailed
	}
	if countData.Total == 0 {
		return &oa.GetPerformanceRankingResp{}, nil
	}
	statistics := make([]*oa.Statistics, 0)
	for _, v := range countData.RankingCountLevelList {
		percent, err := strconv.ParseFloat(fmt.Sprintf("%.2f", float64(v.Count)*100/float64(countData.Total)), 10)
		if err != nil {
			l.Logger.Errorw("failed to parse float on GetPerformanceRanking", logx.Field("error", err))
			return nil, ErrPerformanceQueryFailed
		}
		statistics = append(statistics, &oa.Statistics{Level: v.Level, Count: v.Count, Percent: percent})
	}
	// 自定义排序顺序：A > B+ > B > C > D > E
	sort.Slice(statistics, func(i, j int) bool {
		// 定义优先级映射
		levelOrder := map[string]int{
			"A":  6,
			"B+": 5,
			"B":  4,
			"C":  3,
			"D":  2,
			"E":  1,
		}
		// 按优先级排序
		return levelOrder[statistics[i].Level] > levelOrder[statistics[j].Level]
	})
	q := performance.FindAllByUserAndTimeQuery{
		Year:                in.GetYear(),
		MonthNum:            in.GetMonthNum(),
		SortField:           in.GetSortField(),
		SortOrder:           in.GetSortOrder(),
		PageNum:             in.GetPageNum(),
		PageSize:            in.GetPageSize(),
		IsExcellentEmployee: in.GetIsExcellentEmployee(),
		EmailList:           emailList,
		Status:              in.GetStatus(),
	}
	pList, err = l.svcCtx.PerformanceModel.FindAllByUserAndTime(l.ctx, q)
	if err != nil {
		l.Logger.Errorw("failed to query x_performance on GetPerformanceRanking.FindAllByUserAndTime", logx.Field("error", err))
		return nil, ErrPerformanceQueryFailed
	}

	workItemList, err = l.svcCtx.PerformanceWorkModel.GetWorkItemsByPerformanceIds(l.ctx, pList.Ids())
	if err != nil {
		l.Logger.Errorw("failed to query performance_work_item on GetPerformanceRanking.GetWorkItemsByPerformanceIds", logx.Field("error", err))
		return nil, ErrPerformanceWorkItemQueryFailed
	}

	// 组装
	rankingList := make([]*oa.PerformanceRanking, len(pList))
	for i, p := range pList {
		// 组装工作项
		wList := workItemList.ByPerformanceId(p.Id)
		ws := make([]*oa.PerformanceWorkItem, len(wList))
		for j, w := range wList {
			ws[j] = &oa.PerformanceWorkItem{
				Id:            w.Id,
				ItemName:      w.ItemName,
				ItemWeight:    w.ItemWeight,
				ItemDesc:      w.ItemDesc,
				ItemDeficency: w.ItemDeficiency,
				SelfScore:     w.SelfScore,
				LeaderScore:   w.LeaderScore,
			}
		}

		// 绩效排名
		rankingList[i] = &oa.PerformanceRanking{
			PerformanceId:           p.Id,
			Year:                    p.Year,
			MonthNum:                p.MonthNum,
			UserEmail:               extractEmailUsername(p.UserEmail),
			Email:                   p.UserEmail,
			TotalScore:              p.TotalScore,
			TimeFactor:              p.TimeFactor,
			Level:                   p.Level,
			LevelFactor:             p.LevelFactor,
			IncidentDeduction:       p.IncidentDeduction,
			ExcellentMemberBonus:    p.ExcellentMemberBonus,
			ExcellentManagerBonus:   p.ExcellentManagerBonus,
			IsExcellentEmployee:     p.IsExcellentEmployee,
			ExcellentEmployeeReason: p.ExcellentEmployeeReason,
			TotalBonus:              p.TotalBonus,
			WorkItems:               ws,
			Status:                  oa.PerformanceStatus(p.Status),
			CreatedAt:               timestamppb.New(p.CreatedAt),
			UpdatedAt:               timestamppb.New(p.UpdatedAt),
		}
	}

	resp := &oa.GetPerformanceRankingResp{
		Total:      countData.Total,
		TotalBonus: countData.TotalBonus,
		List:       rankingList,
		Statistics: statistics,
	}

	return resp, nil
}

func (l *GetPerformanceRankingLogic) getPerformanceRankingForOneUser(in *oa.GetPerformanceRankingReq) (*oa.GetPerformanceRankingResp, error) {
	//p, err := l.svcCtx.PerformanceModel.GetOneByUserAndTime(l.ctx, in.GetSearchUserEmail(), in.GetYear(), in.GetMonthNum())//不用这个，其他地方也在用这个方法
	//p, err := l.svcCtx.PerformanceModel.GetOne(l.ctx, in.GetSearchUserEmail(), in.GetYear(), in.GetMonthNum(), in.GetIsExcellentEmployee())
	inSql := performance.FindAllByUserAndTimeQuery{
		Year:                in.GetYear(),
		MonthNum:            in.GetMonthNum(),
		EmailList:           []string{in.GetSearchUserEmail()},
		IsExcellentEmployee: in.GetIsExcellentEmployee(),
		Status:              in.GetStatus(),
	}
	pList, err := l.svcCtx.PerformanceModel.FindAllByUserAndTime(l.ctx, inSql)
	if len(pList) == 0 {
		return &oa.GetPerformanceRankingResp{}, nil
	}
	p := pList[0]
	if err != nil {
		if errors.Is(err, performance.ErrNotFound) {
			return &oa.GetPerformanceRankingResp{}, nil
		}
		l.Logger.Errorw("failed to query x_performance on GetPerformanceRanking.GetOneByUserAndTime", logx.Field("error", err))
		return nil, ErrPerformanceQueryFailed
	}
	workItemList, err := l.svcCtx.PerformanceWorkModel.GetWorkItemsByPerformanceId(l.ctx, p.Id)
	if err != nil {
		l.Logger.Errorw("failed to query x_performance_work on GetPerformanceRanking.GetWorkItemsByPerformanceId", logx.Field("error", err))
		return nil, ErrPerformanceWorkItemQueryFailed
	}

	rankingData := &oa.PerformanceRanking{
		PerformanceId:           p.Id,
		Year:                    p.Year,
		MonthNum:                p.MonthNum,
		UserEmail:               extractEmailUsername(p.UserEmail),
		TotalScore:              p.TotalScore,
		TimeFactor:              p.TimeFactor,
		Level:                   p.Level,
		LevelFactor:             p.LevelFactor,
		IncidentDeduction:       p.IncidentDeduction,
		ExcellentMemberBonus:    p.ExcellentMemberBonus,
		ExcellentManagerBonus:   p.ExcellentManagerBonus,
		TotalBonus:              p.TotalBonus,
		IsExcellentEmployee:     p.IsExcellentEmployee,
		ExcellentEmployeeReason: p.ExcellentEmployeeReason,
		Status:                  oa.PerformanceStatus(p.Status),
		CreatedAt:               timestamppb.New(p.CreatedAt),
		UpdatedAt:               timestamppb.New(p.UpdatedAt),
	}
	workItems := make([]*oa.PerformanceWorkItem, len(workItemList))
	for i, v := range workItemList {
		workItems[i] = &oa.PerformanceWorkItem{
			Id:            v.Id,
			ItemName:      v.ItemName,
			ItemWeight:    v.ItemWeight,
			ItemDesc:      v.ItemDesc,
			ItemDeficency: v.ItemDeficiency,
			SelfScore:     v.SelfScore,
			LeaderScore:   v.LeaderScore,
		}
	}
	rankingData.WorkItems = workItems

	resp := oa.GetPerformanceRankingResp{
		Total:      1,
		TotalBonus: p.TotalBonus,
		List:       []*oa.PerformanceRanking{rankingData},
		Statistics: []*oa.Statistics{
			{
				Level:   p.Level,
				Count:   1,
				Percent: 100,
			},
		},
	}
	return &resp, nil
}
