package performancerpclogic

import (
	"context"
	"errors"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/shopspring/decimal"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type PerformanceLeaderAckLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewPerformanceLeaderAckLogic(ctx context.Context, svcCtx *svc.ServiceContext) *PerformanceLeaderAckLogic {
	return &PerformanceLeaderAckLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 上级确认绩效
func (l *PerformanceLeaderAckLogic) PerformanceLeaderAck(in *oa.PerformanceLeaderAckReq) (*oa.PerformanceLeaderAckResp, error) {
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		l.Logger.Errorw("failed to query table x_performance on PerformanceLeaderAck", logx.Field("error", err), logx.Field("id", in.GetPerformanceId()))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}

	// 查询绩效归属人信息
	user, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on PerformanceLeaderAck", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	if user.LeaderEmail != in.GetUserEmail() {
		return nil, ErrPermissionDenied
	}
	if p.Status == uint64(oa.PerformanceStatus_DONE) {
		return nil, ErrPerformanceCanNotBeModified
	}

	if p.Status == uint64(oa.PerformanceStatus_NEED_LEADER_ACK) {
		p.Status = uint64(oa.PerformanceStatus_DONE)
		// 计算total_bonus
		totalBonus, _ := decimal.NewFromFloat(p.TimeFactor).
			Mul(decimal.NewFromInt(bonusMonthBase)).
			Mul(decimal.NewFromFloat(p.LevelFactor)).
			Sub(decimal.NewFromFloat(p.IncidentDeduction)).
			Add(decimal.NewFromFloat(p.ExcellentMemberBonus)).
			Add(decimal.NewFromFloat(p.ExcellentManagerBonus)).Round(2).Float64()
		// 更新total_bonus
		p.TotalBonus = totalBonus
		if err = l.svcCtx.PerformanceModel.XUpdate(l.ctx, p); err != nil {
			l.Logger.Errorw("failed to update table x_performance on PerformanceLeaderAck", logx.Field("error", err), logx.Field("id", p.Id))
			return nil, ErrPerformanceUpdateFailed
		}
	}

	// 发送lark通知
	//common.AsyncSendLarkMsg(l.svcCtx.Config.Env, func() {
	//	msgData := map[string]any{
	//		"title":      genPerformanceLarkMsgTitle(p.Year, p.MonthNum),
	//		"content":    fmt.Sprintf("%s 已确认绩效结果，请知晓", extractEmailUsername(user.LeaderEmail)),
	//		"target_url": l.svcCtx.Config.Lark.MyPerformanceUrl,
	//	}
	//	msg, err := l.svcCtx.LarkService.BuildCardContent(l.svcCtx.Config.Lark.PerformanceTmpId, msgData)
	//	if err != nil {
	//		l.Logger.Errorw("failed to build lark card content", logx.Field("error", err))
	//		return
	//	}
	//	if err = l.svcCtx.LarkService.SendCardMessage(msgcenter.LarkReceiveIdTypeEmail, p.UserEmail, msgcenter.LarkMsgTypeCard, msg); err != nil {
	//		l.Logger.Errorw("failed to send user lark msg", logx.Field("error", err))
	//		return
	//	}
	//})

	return &oa.PerformanceLeaderAckResp{}, nil
}
