package performancerpclogic

import (
	"context"
	"errors"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type RollbackPerformanceStatusLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewRollbackPerformanceStatusLogic(ctx context.Context, svcCtx *svc.ServiceContext) *RollbackPerformanceStatusLogic {
	return &RollbackPerformanceStatusLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 打回绩效
func (l *RollbackPerformanceStatusLogic) RollbackPerformanceStatus(in *oa.RollbackPerformanceStatusReq) (*oa.RollbackPerformanceStatusResp, error) {
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		l.Logger.Errorw("failed to query x_performance on RollbackPerformanceStatus", logx.Field("id", in.GetPerformanceId()), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}

	// 查询绩效归属人信息
	user, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on RollbackPerformanceStatus", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	// 只有leader可以回滚绩效状态
	if user.LeaderEmail != in.GetUserEmail() {
		return nil, ErrPermissionDenied
	}

	switch p.Status {
	case uint64(oa.PerformanceStatus_NEED_LEADER_ACK), uint64(oa.PerformanceStatus_NEED_STAFF_ACK):
		p.Status = uint64(oa.PerformanceStatus_LEADER_EVALUATION)

	case uint64(oa.PerformanceStatus_LEADER_EVALUATION):
		p.Status = uint64(oa.PerformanceStatus_WORKLOAD_FILLING)
	default:
		return nil, status.Error(codes.PermissionDenied, "当前绩效不支持状态回滚")
	}

	if err = l.svcCtx.PerformanceModel.XUpdate(l.ctx, p); err != nil {
		logx.Errorw("failed to update x_performance on RollbackPerformanceStatus", logx.Field("error", err))
		return nil, ErrPerformanceUpdateFailed
	}

	return &oa.RollbackPerformanceStatusResp{}, nil
}
