package performancerpclogic

import (
	"context"
	"errors"
	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type UpdatePerformanceAdditionItemLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewUpdatePerformanceAdditionItemLogic(ctx context.Context, svcCtx *svc.ServiceContext) *UpdatePerformanceAdditionItemLogic {
	return &UpdatePerformanceAdditionItemLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 修改绩效附加项
func (l *UpdatePerformanceAdditionItemLogic) UpdatePerformanceAdditionItem(in *oa.UpdatePerformanceAdditionItemReq) (*oa.UpdatePerformanceAdditionItemResp, error) {
	// 查询绩效附加项
	additionItem, err := l.svcCtx.PerformanceAddtionModel.XFindOne(l.ctx, in.GetItemId())
	if err != nil {
		l.Logger.Errorw("failed to query table x_performance_addition on UpdatePerformanceAdditionItem", logx.Field("id", in.GetItemId()), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceAdditionItemNotFound
		}
		return nil, ErrPerformanceAdditionItemQueryFailed
	}

	// 根据additionItem.performance_id查询绩效
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, additionItem.PerformanceId)
	if err != nil {
		l.Logger.Errorw("failed to query table x_performance on UpdatePerformanceAdditionItem", logx.Field("id", additionItem.PerformanceId), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}
	//if p.Status >= uint64(oa.PerformanceStatus_NEED_LEADER_ACK) {
	//	// 待上级确认、结束 这两个状态是不允许修改的。
	//	return nil, ErrPerformanceCanNotBeModified
	//}
	if p.Status == uint64(oa.PerformanceStatus_NEED_LEADER_ACK) || p.Status == uint64(oa.PerformanceStatus_DONE) {
		// 待上级确认、结束 这两个状态是不允许修改的  4,5
		return nil, ErrPerformanceCanNotBeModified
	}

	// 查询绩效归属人信息
	user, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on UpdatePerformanceAdditionItem", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	switch in.GetUserEmail() {
	case user.LeaderEmail:
		// 上级打分，可以打0分、满分150
		if err := isLeaderScoreValid(in.GetLeaderScore()); err != nil {
			return nil, err
		}
		additionItem.LeaderScore = in.GetLeaderScore()

		// 上级打分时实时计算绩效总分
		if err = l.leaderScoreAdditionItem(p, additionItem); err != nil {
			return nil, ErrPerformanceAdditionItemUpdateFailed
		}
	case p.UserEmail:
		// 员工更新绩效附加项。自评分不可为0分
		if additionItem.ItemName == additionItemCoworker {
			if additionItem.SelfScore != 0 { //已经被其他部门同事评价过了
				return &oa.UpdatePerformanceAdditionItemResp{}, nil
			}
			if in.GetCoworkerEmail() == "" || in.GetCoworkerEmail() == in.GetUserEmail() { //被邀请人不能是当前登录的用户
				return nil, status.Error(codes.InvalidArgument, "参数无效")
			}
			additionItem.CoworkerEmail = in.GetCoworkerEmail()
		} else {
			if err := isSelfScoreValid(in.GetSelfScore()); err != nil {
				return nil, err
			} //当不是其他部门同事评价的时候（制度，生产bug）需要判断评分是否符合要求，如果是其他部门同事评价，自己不打分，所以也就没有这个判断
			additionItem.CoworkerEmail = in.GetCoworkerEmail()
			additionItem.ItemDesc = in.GetItemDesc()
			additionItem.SelfScore = in.GetSelfScore()
		}
		if err = l.svcCtx.PerformanceAddtionModel.XUpdate(l.ctx, additionItem); err != nil {
			l.Logger.Errorw("failed to update x_performance_addition itemDesc and selfScore", logx.Field("error", err), logx.Field("id", additionItem.Id))
			return nil, ErrPerformanceAdditionItemUpdateFailed
		}

	default:
		// 其他人无权限操作
		return nil, ErrPermissionDenied
	}

	return &oa.UpdatePerformanceAdditionItemResp{}, nil
}

func (l *UpdatePerformanceAdditionItemLogic) leaderScoreAdditionItem(p *performance.XPerformance, additionItem *performance.XPerformanceAddition) error {
	return l.svcCtx.PerformanceWorkModel.TransCtx(l.ctx, func(ctx context.Context, s sqlx.Session) error {
		// 更新performance_work分数
		if err := l.svcCtx.PerformanceAddtionModel.TransUpdateCtx(ctx, s, additionItem); err != nil {
			logx.Errorw("failed to update x_performance_work on UpdatePerformanceAdditionItem by leader", logx.Field("error", err))
			return err
		}
		// 查询工作项
		workItemList, err := l.svcCtx.PerformanceWorkModel.TransGetWorkItemsByPerformanceId(ctx, s, p.Id)
		if err != nil {
			logx.Errorw("failed to query x_performance_work on UpdatePerformanceAdditionItem by leader", logx.Field("error", err))
			return err
		}
		// 查询附加工作项
		additionItemList, err := l.svcCtx.PerformanceAddtionModel.TransGetAdditionItemsByPerformandId(ctx, s, p.Id)
		if err != nil {
			logx.Errorw("failed to query x_performance_addition on UpdatePerformanceAdditionItem by leader", logx.Field("error", err))
			return err
		}

		// 计算绩效总分, 并更新绩效状态、总分
		p.TotalScore = caculatePerformanceScore(workItemList, additionItemList)
		level, levelFactor := getPerformanceLevelByScore(p.TotalScore)
		p.Level = level
		p.LevelFactor = levelFactor

		// 更新绩效总分
		if err = l.svcCtx.PerformanceModel.TransUpdateCtx(ctx, s, p); err != nil {
			logx.Errorw("failed to update x_performance on UpdatePerformanceAdditionItem by leader", logx.Field("error", err))
			return err
		}
		return nil
	})
}
