package performancerpclogic

import (
	"context"
	"errors"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/globalkey"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/shopspring/decimal"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type UpdatePerformanceBonusLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewUpdatePerformanceBonusLogic(ctx context.Context, svcCtx *svc.ServiceContext) *UpdatePerformanceBonusLogic {
	return &UpdatePerformanceBonusLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 更新绩效奖金
func (l *UpdatePerformanceBonusLogic) UpdatePerformanceBonus(in *oa.UpdatePerformanceBonusReq) (*oa.UpdatePerformanceBonusResp, error) {
	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, in.GetPerformanceId())
	if err != nil {
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		l.Logger.Errorw("failed to query x_performnace on UpdatePerformanceBonus", logx.Field("error", err))
		return nil, ErrPerformanceQueryFailed
	}

	if p.Status != uint64(oa.PerformanceStatus_DONE) && p.Status != uint64(oa.PerformanceStatus_NEED_CONFIRM) && p.Status != uint64(oa.PerformanceStatus_SUBMITTED) {
		return nil, status.Error(codes.PermissionDenied, "当前绩效正在进行中，无法修改")
	}

	// 查询绩效归属人信息
	user, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on UpdatePerformanceBonus", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	if user.LeaderEmail != in.GetUserEmail() && in.GetUserEmail() != globalkey.RockyEmail {
		return nil, ErrPermissionDenied
	}
	excellentEmployeeReason := in.GetExcellentEmployeeReason()
	if in.GetIsExcellentEmployee() == uint64(oa.IsExcellentEmployee_IS_EXCELLENT_EMPLOYEE_NO) { // 不是优秀员工
		excellentEmployeeReason = ""
	}

	data := performance.UpdatePerformanceBonusQuery{
		PerformanceID:           in.GetPerformanceId(),
		TimeFactor:              in.GetTimeFactor(),
		LevelFactor:             in.GetLevelFactor(),
		IncidentDeduction:       in.GetIncidentDeduction(),
		ExcellentMemberBonus:    in.GetExcellentMemberBonus(),
		ExcellentManagerBonus:   in.GetExcellentManagerBonus(),
		IsExcellentEmployee:     in.GetIsExcellentEmployee(),
		ExcellentEmployeeReason: excellentEmployeeReason,
	}
	// 计算total_bonus
	totalBonus, _ := decimal.NewFromFloat(data.TimeFactor).
		Mul(decimal.NewFromInt(bonusMonthBase)).
		Mul(decimal.NewFromFloat(data.LevelFactor)).
		Sub(decimal.NewFromFloat(data.IncidentDeduction)).
		Add(decimal.NewFromFloat(data.ExcellentMemberBonus)).
		Add(decimal.NewFromFloat(data.ExcellentManagerBonus)).Round(2).Float64()
	// 更新total_bonus
	data.TotalBonus = totalBonus

	if err = l.svcCtx.PerformanceModel.UpdatePerformanceBonus(l.ctx, data); err != nil {
		l.Logger.Errorw("failed to update x_performance on UpdatePerformanceBonus", logx.Field("error", err))
		return nil, ErrPerformanceUpdateFailed
	}

	return &oa.UpdatePerformanceBonusResp{}, nil
}
