package performancerpclogic

import (
	"context"
	"errors"
	"fmt"

	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/shopspring/decimal"
	"github.com/zeromicro/go-zero/core/logx"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type UpdatePerformanceWorkItemLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewUpdatePerformanceWorkItemLogic(ctx context.Context, svcCtx *svc.ServiceContext) *UpdatePerformanceWorkItemLogic {
	return &UpdatePerformanceWorkItemLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 修改绩效工作部分
func (l *UpdatePerformanceWorkItemLogic) UpdatePerformanceWorkItem(in *oa.UpdatePerformanceWorkItemReq) (*oa.UpdatePerformanceWorkItemResp, error) {
	workItem, err := l.svcCtx.PerformanceWorkModel.XFindOne(l.ctx, in.GetItemId())
	if err != nil {
		l.Logger.Errorw("failed to query x_performance_work on UpdatePerformancWorkItem", logx.Field("id", in.GetItemId()), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceWorkItemNotFound
		}
		return nil, ErrPerformanceWorkItemQueryFailed
	}

	p, err := l.svcCtx.PerformanceModel.XFindOne(l.ctx, workItem.PerformanceId)
	if err != nil {
		l.Logger.Errorw("failed to query x_performance on UpdatePerformancWorkItem", logx.Field("id", workItem.PerformanceId), logx.Field("error", err))
		if errors.Is(err, performance.ErrNotFound) {
			return nil, ErrPerformanceNotFound
		}
		return nil, ErrPerformanceQueryFailed
	}

	if p.Status == uint64(oa.PerformanceStatus_NEED_LEADER_ACK) || p.Status == uint64(oa.PerformanceStatus_DONE) {
		// 待上级确认、结束 这两个状态是不允许修改的  4,5
		return nil, ErrPerformanceCanNotBeModified
	}

	// 查询绩效归属人信息
	user, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: p.UserEmail})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on UpdatePerformanceWorkItem", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	switch in.GetUserEmail() {
	case user.LeaderEmail:
		// 上级打分，可以打0分、满分150
		if err := isLeaderScoreValid(in.GetLeaderScore()); err != nil {
			return nil, err
		}
		workItem.LeaderScore = in.GetLeaderScore()

		// 上级打分，需要实时计算分数
		if err = l.leaderScoreWorktItem(p, workItem); err != nil {
			return nil, ErrPerformanceWorkItemUpdateFailed
		}

	case p.UserEmail:
		// 员工更新绩效工作项
		if err := isItemWeightValid(in.GetItemWeight()); err != nil {
			return nil, err
		}

		if err := isSelfScoreValid(in.GetSelfScore()); err != nil {
			return nil, err
		}

		// 计算工作部分权重和是否超过指定范围
		// 管理人员 70%  普通员工 75%
		workItems, err := l.svcCtx.PerformanceWorkModel.GetWorkItemsByPerformanceId(l.ctx, p.Id)
		if err != nil {
			l.Logger.Errorw("failed to query table x_performance_work on UpdatePerformancWorkItem", logx.Field("error", err), logx.Field("performance_id", p.Id))
			return nil, ErrPerformanceWorkItemQueryFailed
		}

		workItemWeightOld := decimal.NewFromFloat(workItem.ItemWeight)
		workItemWeightNew := decimal.NewFromFloat(in.GetItemWeight())
		weightTotal := sumWorkItemWeight(workItems)
		//if len(user.OwnerDepartmentList) > 0 {
		//	// 领导工作部分权重和为70%
		//	if weightTotal.Sub(workItemWeightOld).Add(workItemWeightNew).GreaterThan(decimal.NewFromFloat(managerWorkWeightTotal)) {
		//		return nil, status.Error(codes.InvalidArgument, "工作部分权重和最高为70%")
		//	}
		//} else {
		//	// 普通员工工作部分权重和为75%
		//	if weightTotal.Sub(workItemWeightOld).Add(workItemWeightNew).GreaterThan(decimal.NewFromFloat(staffWorkWeightTotal)) {
		//		return nil, status.Error(codes.InvalidArgument, "工作部分权重和最高为75%")
		//	}
		//}
		if weightTotal.Sub(workItemWeightOld).Add(workItemWeightNew).GreaterThan(decimal.NewFromFloat(p.WorkItemWeight)) {
			return nil, status.Error(codes.InvalidArgument, fmt.Sprintf("工作部分权重和最高为%d%%", int32(p.WorkItemWeight)))
		}
		workItem.ItemName = in.GetItemName()
		workItem.ItemWeight = in.GetItemWeight()
		workItem.ItemDesc = in.GetItemDesc()
		workItem.ItemDeficiency = in.GetItemDeficency()
		workItem.SelfScore = in.GetSelfScore()

		if err = l.svcCtx.PerformanceWorkModel.XUpdate(l.ctx, workItem); err != nil {
			l.Logger.Errorw("failed to update table x_performance_work on UpdatePerformanceWorkItem", logx.Field("error", err), logx.Field("id", workItem.Id))
			return nil, ErrPerformanceWorkItemUpdateFailed
		}
	default:
		return nil, ErrPermissionDenied
	}

	return &oa.UpdatePerformanceWorkItemResp{}, nil
}

func (l *UpdatePerformanceWorkItemLogic) leaderScoreWorktItem(p *performance.XPerformance, workItem *performance.XPerformanceWork) error {
	return l.svcCtx.PerformanceWorkModel.TransCtx(l.ctx, func(ctx context.Context, s sqlx.Session) error {
		// 更新performance_work分数
		if err := l.svcCtx.PerformanceWorkModel.TransUpdateCtx(ctx, s, workItem); err != nil {
			logx.Errorw("failed to update x_performance_work on UpdatePerformanceWorkItem by leader", logx.Field("error", err))
			return err
		}
		// 查询工作项
		workItemList, err := l.svcCtx.PerformanceWorkModel.TransGetWorkItemsByPerformanceId(ctx, s, p.Id)
		if err != nil {
			logx.Errorw("failed to query x_performance_work on UpdatePerformanceWorkItem by leader", logx.Field("error", err))
			return err
		}
		// 查询附加工作项
		additionItemList, err := l.svcCtx.PerformanceAddtionModel.TransGetAdditionItemsByPerformandId(ctx, s, p.Id)
		if err != nil {
			logx.Errorw("failed to query x_performance_addition on UpdatePerformanceWorkItem by leader", logx.Field("error", err))
			return err
		}

		// 计算绩效总分, 并更新绩效状态、总分
		p.TotalScore = caculatePerformanceScore(workItemList, additionItemList)
		level, levelFactor := getPerformanceLevelByScore(p.TotalScore)
		p.Level = level
		p.LevelFactor = levelFactor

		// 更新绩效总分
		if err = l.svcCtx.PerformanceModel.TransUpdateCtx(ctx, s, p); err != nil {
			logx.Errorw("failed to update x_performance on UpdatePerformanceWorkItem by leader", logx.Field("error", err))
			return err
		}
		return nil
	})
}
