package reportrpclogic

import (
	"context"
	"errors"
	"fmt"

	"oa-server/app/oacenter/model/report"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type CreateReportCommentLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewCreateReportCommentLogic(ctx context.Context, svcCtx *svc.ServiceContext) *CreateReportCommentLogic {
	return &CreateReportCommentLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 新建评论
func (l *CreateReportCommentLogic) CreateReportComment(in *oa.CreateReportCommentReq) (*oa.CreateReportCommentResp, error) {
	if err := l.Validate(in); err != nil {
		return nil, err
	}

	rpt, err := l.svcCtx.ReportModel.FindOne(l.ctx, in.GetReportId())
	if err != nil {
		if errors.Is(err, report.ErrNotFound) {
			return nil, ErrReportNotFound
		}
		logx.Errorw("failed to query x_report on CreateReportComment.ReportModel.FindOne", logx.Field("error", err))
		return nil, ErrReportQueryFailed
	}

	// 每个工作汇报最多只能有5条评论
	reportComments, err := l.svcCtx.ReportCommentModel.FindByReportIds(l.ctx, []uint64{in.GetReportId()})
	if err != nil {
		logx.Errorw("failed to query x_report_comment on CreateReportComment.ReportCommentModel.FindByReportIds", logx.Field("error", err))
		return nil, ErrReportCommentQueryFailed
	}
	if len(reportComments) >= ReportCommentMaxCount {
		return nil, status.Error(codes.PermissionDenied, fmt.Sprintf("每个工作汇报最多只能有%d个评论", ReportCommentMaxCount))
	}

	comment := &report.XReportComment{
		ReportId:  rpt.Id,
		UserEmail: in.GetUserEmail(),
		Content:   in.GetContent(),
	}
	result, err := l.svcCtx.ReportCommentModel.Insert(l.ctx, comment)
	if err != nil {
		logx.Errorw("failed to create x_report_comment on CreateReportComment.ReportCommentModel.Insert", logx.Field("error", err))
		return nil, ErrReportCommentCreateFailed
	}
	commentId, err := result.LastInsertId()
	if err != nil {
		logx.Errorw("failed to get x_report_comment LastInsertId", logx.Field("error", err))
		return nil, ErrReportCommentCreateFailed
	}

	return &oa.CreateReportCommentResp{ReportCommentId: uint64(commentId)}, nil
}

func (l *CreateReportCommentLogic) Validate(in *oa.CreateReportCommentReq) error {
	if in.GetUserEmail() == "" {
		return ErrUserEmailRequired
	}
	if in.GetReportId() == 0 {
		return ErrReportIdRequired
	}
	if in.GetContent() == "" {
		return status.Error(codes.InvalidArgument, "评论内容不能为空")
	}
	return nil
}
