package reportrpclogic

import (
	"context"
	"fmt"
	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"golang.org/x/time/rate"
	"maps"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/globalkey"
	"oa-server/common/msgcenter"
	"slices"
	"strings"
	"time"
)

type GetReportStatisticsLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetReportStatisticsLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetReportStatisticsLogic {
	return &GetReportStatisticsLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取部门工作汇报统计
func (l *GetReportStatisticsLogic) GetReportStatistics(in *oa.ReportStatisticsReq) (*oa.ReportStatisticsResp, error) {
	reportStartDateTime, err := time.ParseInLocation(time.DateOnly, in.ReportStartDate, time.Local)
	if err != nil {
		return nil, fmt.Errorf("parse reportStartDate err: %s", err.Error())
	}
	reportEndDateTime, err := time.ParseInLocation(time.DateOnly, in.ReportEndDate, time.Local)
	if err != nil {
		return nil, fmt.Errorf("parse reportEndDate err: %s", err.Error())
	}
	if reportStartDateTime.After(reportEndDateTime) || reportStartDateTime.AddDate(0, 0, 62).Before(reportEndDateTime) {
		return nil, fmt.Errorf("invalid reportStartDate reportEndDate pair")
	}

	rtn := &oa.ReportStatisticsResp{
		Departments: &oa.Departments{
			Level3: nil,
			Level4: nil,
			Level5: nil,
		},
	}

	publicUserDepartmentListWithChildrenReq := usercenter.PublicUserDepartmentListWithChildrenReq{
		AppTokenMustModel: usercenter.AppTokenMustModel{
			AppKey:   l.svcCtx.Config.LbkUsercenter.AppName,
			AppToken: l.svcCtx.Config.LbkUsercenter.AppToken,
		},
		Owner: in.GetUserEmail(),
		//Owner: "wangming.xie@lbk.one",
		//Owner: globalkey.RockyEmail,
	}
	managedDepartments, err := l.svcCtx.LbkUcClient.GetDepartmentListWithChildren(l.ctx, publicUserDepartmentListWithChildrenReq)
	if err != nil {
		logx.Errorw("failed to call usercenter.GetDepartmentListWithChildren on GetReportStatistics", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}
	managedDepartmentToMembers := make(map[string]map[string]struct{})
	for _, department := range managedDepartments {
		managedDepartmentToMembers[department.DepartmentKey] = make(map[string]struct{})
	}
	allDepartments := slices.Collect(maps.Keys(managedDepartmentToMembers))
	if len(allDepartments) == 0 {
		return rtn, nil
	}

	getDepartmentDirectlyUserReq := usercenter.GetDepartmentDirectlyUserReq{
		AppKey:           l.svcCtx.Config.LbkUsercenter.AppName,
		AppToken:         l.svcCtx.Config.LbkUsercenter.AppToken,
		DeparmentKeyList: allDepartments,
	}

	followers, err := l.svcCtx.LbkUcClient.GetDepartmentDirectlyUser(l.ctx, getDepartmentDirectlyUserReq)
	if err != nil {
		logx.Errorw("failed to call usercenter.GetDepartmentDirectlyUser on GetReportStatistics", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	for _, follower := range followers {
		for _, dk := range follower.DepartmentKeyList { //最具体的部门
			if _, ok := managedDepartmentToMembers[dk]; ok {
				managedDepartmentToMembers[dk][follower.Email] = struct{}{}
			}
		}
		for _, sd := range follower.SecondDepartmentList { //二级部门（如效能部）
			if _, ok := managedDepartmentToMembers[sd.DepartmentKey]; ok {
				managedDepartmentToMembers[sd.DepartmentKey][follower.Email] = struct{}{}
			}
		}
	}

	temp := make(map[string]struct{})
	for _, members := range managedDepartmentToMembers {
		maps.Copy(temp, members)
	}
	allUsers := slices.Collect(maps.Keys(temp))

	individualTypeToCount := make(map[string]map[string]int32)
	defaultTypeToCount := map[string]int32{
		"1": 0,
		"2": 0,
		"3": 0,
	}
	for _, user := range allUsers {
		individualTypeToCount[user] = maps.Clone(defaultTypeToCount)
	}

	defaultDateToCount := make(map[string]int32)
	for {
		day := reportStartDateTime.Local().Format(time.DateOnly)
		defaultDateToCount[day] = 0
		reportStartDateTime = reportStartDateTime.AddDate(0, 0, 1)
		if reportStartDateTime.After(reportEndDateTime) {
			break
		}
	}
	individualDateToCount := make(map[string]map[string]int32)
	for _, user := range allUsers {
		individualDateToCount[user] = maps.Clone(defaultDateToCount)
	}

	listByType, err := l.svcCtx.ReportModel.GetReportCountByUserAndType(l.ctx, allUsers, in.GetReportStartDate(), in.GetReportEndDate())
	if err != nil {
		return nil, err
	}
	for _, v := range listByType {
		individualTypeToCount[v.UserEmail][fmt.Sprint(v.ReportType)] = int32(v.Count)
	}

	listByDate, err := l.svcCtx.ReportModel.GetReportCountByUserAndDate(l.ctx, allUsers, in.GetReportStartDate(), in.GetReportEndDate())
	if err != nil {
		return nil, err
	}
	for _, v := range listByDate {
		timeDate, _ := time.ParseInLocation(time.RFC3339, v.ReportDate, time.Local)
		individualDateToCount[v.UserEmail][timeDate.Local().Format(time.DateOnly)] = int32(v.Count)
	}

	for _, department := range managedDepartments {
		departmentStatistics := &oa.DepartmentStatistics{
			DepartmentKey:     department.DepartmentKey,
			DepartmentName:    department.DepartmentName,
			ReportTypeToCount: make(map[string]int32),
			ReportDateToCount: make(map[string]int32),
			Individuals:       nil,
		}
		for member := range managedDepartmentToMembers[department.DepartmentKey] {
			for type_, count := range individualTypeToCount[member] {
				departmentStatistics.ReportTypeToCount[type_] += count
			}
			for date_, count := range individualDateToCount[member] {
				departmentStatistics.ReportDateToCount[date_] += count
			}
			departmentStatistics.Individuals = append(departmentStatistics.Individuals, &oa.IndividualStatistics{
				UserEmail:         member,
				ReportTypeToCount: individualTypeToCount[member],
				ReportDateToCount: individualDateToCount[member],
			})
		}
		slices.SortFunc(departmentStatistics.Individuals, func(a, b *oa.IndividualStatistics) int {
			return -strings.Compare(a.UserEmail, b.UserEmail)
		})
		if department.DepartmentLevel == 3 {
			rtn.Departments.Level3 = append(rtn.Departments.Level3, departmentStatistics)
		}
		if department.DepartmentLevel == 4 {
			rtn.Departments.Level4 = append(rtn.Departments.Level4, departmentStatistics)
		}
		if department.DepartmentLevel == 5 {
			rtn.Departments.Level5 = append(rtn.Departments.Level5, departmentStatistics)
		}
	}
	slices.SortFunc(rtn.Departments.Level3, func(a, b *oa.DepartmentStatistics) int {
		return -strings.Compare(a.DepartmentName, b.DepartmentName)
	})
	slices.SortFunc(rtn.Departments.Level4, func(a, b *oa.DepartmentStatistics) int {
		return -strings.Compare(a.DepartmentName, b.DepartmentName)
	})
	slices.SortFunc(rtn.Departments.Level5, func(a, b *oa.DepartmentStatistics) int {
		return -strings.Compare(a.DepartmentName, b.DepartmentName)
	})

	return rtn, nil
}

func (l *GetReportStatisticsLogic) CheckAndRemindTodayReport() {
	var allUserCount int64
	var sentUserCount int64
	var ignoredUserCount int64
	var failedUserCount int64
	var errorTypeCount = map[string]int64{}
	var err error

	defer func() {
		logx.Infow("[ReportAlertBot] ", logx.Field("allUserCount", allUserCount))
		logx.Infow("[ReportAlertBot] ", logx.Field("sentUserCount", sentUserCount))
		logx.Infow("[ReportAlertBot] ", logx.Field("ignoredUserCount", ignoredUserCount))
		logx.Infow("[ReportAlertBot] ", logx.Field("failedUserCount", failedUserCount))
		logx.Infow("[ReportAlertBot] ", logx.Field("errorTypeCount", fmt.Sprintf("%#v", errorTypeCount)))
		logx.Infow("[ReportAlertBot] ", logx.Field("err", err))
	}()

	publicUserDepartmentListWithChildrenReq := usercenter.PublicUserDepartmentListWithChildrenReq{
		AppTokenMustModel: usercenter.AppTokenMustModel{
			AppKey:   l.svcCtx.Config.LbkUsercenter.AppName,
			AppToken: l.svcCtx.Config.LbkUsercenter.AppToken,
		},
		Owner: globalkey.RockyEmail,
	}
	managedDepartments, err := l.svcCtx.LbkUcClient.GetDepartmentListWithChildren(l.ctx, publicUserDepartmentListWithChildrenReq)
	if err != nil {
		logx.Errorw("[ReportAlertBot] failed to call usercenter.GetDepartmentListWithChildren on CheckAndRemindTodayReport", logx.Field("error", err))
		return
	}
	managedDepartmentToMembers := make(map[string]map[string]struct{})
	for _, department := range managedDepartments {
		managedDepartmentToMembers[department.DepartmentKey] = make(map[string]struct{})
	}
	getDepartmentDirectlyUserReq := usercenter.GetDepartmentDirectlyUserReq{
		AppKey:           l.svcCtx.Config.LbkUsercenter.AppName,
		AppToken:         l.svcCtx.Config.LbkUsercenter.AppToken,
		DeparmentKeyList: slices.Collect(maps.Keys(managedDepartmentToMembers)),
	}

	followers, err := l.svcCtx.LbkUcClient.GetDepartmentDirectlyUser(l.ctx, getDepartmentDirectlyUserReq)
	if err != nil {
		logx.Errorw("[ReportAlertBot] failed to call usercenter.GetDepartmentDirectlyUser on CheckAndRemindTodayReport", logx.Field("error", err))
		return
	}

	for _, follower := range followers {
		for _, dk := range follower.DepartmentKeyList { //最具体的部门
			if _, ok := managedDepartmentToMembers[dk]; ok {
				managedDepartmentToMembers[dk][follower.Email] = struct{}{}
			}
		}
		for _, sd := range follower.SecondDepartmentList { //二级部门（如效能部）
			if _, ok := managedDepartmentToMembers[sd.DepartmentKey]; ok {
				managedDepartmentToMembers[sd.DepartmentKey][follower.Email] = struct{}{}
			}
		}
	}

	temp := make(map[string]struct{})
	for _, members := range managedDepartmentToMembers {
		maps.Copy(temp, members)
	}
	allUsers := slices.Collect(maps.Keys(temp))

	today := time.Now().Local().Format(time.DateOnly)

	// 创建一个限流器，每秒1个令牌，桶大小为1
	// rate.Limit(1) 表示每秒1个令牌
	// 1 表示桶的容量，这里设置为1表示突发不超过1次
	limiter := rate.NewLimiter(rate.Limit(1), 1)

	for _, user := range allUsers {
		if user == globalkey.RockyEmail {
			continue
		}
		allUserCount++
		listByDate, err := l.svcCtx.ReportModel.GetReportCountByUserAndDate(l.ctx, []string{user}, today, today)
		if err != nil {
			logx.Errorw("[ReportAlertBot] failed to call ReportModel.GetReportCountByUserAndDate on CheckAndRemindTodayReport ", logx.Field("error", err), logx.Field("user", user))
			failedUserCount++
			errorTypeCount[fmt.Sprintf("%#v", err)] += 1
			continue
		}
		if len(listByDate) == 0 {

			// 等待获取令牌，如果获取不到会阻塞
			err := limiter.Wait(context.Background())
			if err != nil {
				l.Logger.Errorw("[ReportAlertBot] limiter wait error", logx.Field("error", err))
			}

			// 获取到令牌后
			if l.svcCtx.Config.Env == common.OA_RUNTIME_PROD_ENV { // 非生产环境不发消息
				msg, err := l.svcCtx.LarkService.BuildCardContent(l.svcCtx.Config.Lark.ReportRemindTmpId, nil)
				if err != nil {
					l.Logger.Errorw("[ReportAlertBot] failed to build lark card content", logx.Field("error", err), logx.Field("user", user))
					failedUserCount++
					errorTypeCount[fmt.Sprintf("%#v", err)] += 1
					continue
				}
				if err = l.svcCtx.LarkService.SendCardMessage(msgcenter.LarkReceiveIdTypeEmail, user, msgcenter.LarkMsgTypeCard, msg); err != nil {
					l.Logger.Errorw("[ReportAlertBot] failed to send user lark msg", logx.Field("error", err), logx.Field("user", user))
					failedUserCount++
					errorTypeCount[fmt.Sprintf("%#v", err)] += 1
					continue
				}
			}

			sentUserCount++
		} else {
			ignoredUserCount++
		}
	}
}
