package reportrpclogic

import (
	"context"
	"fmt"
	"golang.org/x/time/rate"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/common/msgcenter"
	"oa-server/common/tool"
	"strings"
	"time"

	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
)

type InformToWriteTodayReportLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewInformToWriteTodayReportLogic(ctx context.Context, svcCtx *svc.ServiceContext) *InformToWriteTodayReportLogic {
	return &InformToWriteTodayReportLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

var InformTodayReportLimiter = rate.NewLimiter(rate.Limit(10), 1)

// 提醒填写今日日报
func (l *InformToWriteTodayReportLogic) InformToWriteTodayReport(in *oa.InformToWriteTodayReportReq) (*oa.InformToWriteTodayReportResp, error) {
	tool.GoSafe(func() {
		today := time.Now().Local().Format(time.DateOnly)
		if in.ReportDate != "" {
			reportDate, err := time.ParseInLocation(time.DateOnly, in.ReportDate, time.Local)
			if err != nil {
				return
			}
			today = reportDate.Format(time.DateOnly)
		}

		for _, user := range in.UserEmails {
			listByDate, err := l.svcCtx.ReportModel.GetReportCountByUserAndDate(context.Background(), []string{user}, today, today)
			if err != nil {
				logx.Errorw("[ReportAlertBot] failed to call ReportModel.GetReportCountByUserAndDate on CheckAndRemindTodayReport ", logx.Field("error", err), logx.Field("user", user))
				continue
			}
			if len(listByDate) == 0 {

				// 等待获取令牌，如果获取不到会阻塞
				err := InformTodayReportLimiter.Wait(context.Background())
				if err != nil {
					l.Logger.Errorw("[ReportAlertBot] limiter wait error", logx.Field("error", err))
				}

				// 获取到令牌后
				if l.svcCtx.Config.Env == common.OA_RUNTIME_PROD_ENV { // 非生产环境不发消息
					//if true { // 非生产环境不发消息
					msg, err := l.svcCtx.LarkService.BuildCardContent(
						l.svcCtx.Config.Lark.GenericPlainTextTmpId,
						map[string]any{
							"title":           "工作汇报提醒",
							"body":            fmt.Sprintf("%s提醒您填写%s工作汇报～", extractEmailUsername(in.SingedInUser), today),
							"action_btn":      "现在去写",
							"action_btn_link": "https://oa.lbkwork.com/workReport/report",
						})
					if err != nil {
						l.Logger.Errorw("[ReportAlertBot] failed to build lark card content", logx.Field("error", err), logx.Field("user", user))
						continue
					}
					if err = l.svcCtx.LarkService.SendCardMessage(msgcenter.LarkReceiveIdTypeEmail, user, msgcenter.LarkMsgTypeCard, msg); err != nil {
						l.Logger.Errorw("[ReportAlertBot] failed to send user lark msg", logx.Field("error", err), logx.Field("user", user))
						continue
					}
				}
			}
		}
	})
	return &oa.InformToWriteTodayReportResp{}, nil
}

func extractEmailUsername(email string) string {
	if email == "" {
		return email
	}

	ret := strings.Split(email, "@")
	return ret[0]
}
