package reportrpclogic

import (
	"context"
	"golang.org/x/text/collate"
	"golang.org/x/text/language"
	"sort"
	"time"

	"oa-server/app/oacenter/model/report"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/globalkey"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type ListDepartmentReportsLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewListDepartmentReportsLogic(ctx context.Context, svcCtx *svc.ServiceContext) *ListDepartmentReportsLogic {
	return &ListDepartmentReportsLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取部门工作汇报列表
func (l *ListDepartmentReportsLogic) ListDepartmentReports(in *oa.ListDepartmentReportsReq) (*oa.ListDepartmentReportsResp, error) {
	reportDate, err := l.Validate(in)
	if err != nil {
		return nil, err
	}

	// 校验用户是否有权限查询对应部门的数据
	userInfo, err := l.svcCtx.LbkUcClient.GetLbkUserDetail(l.ctx, usercenter.LbkUserDetailReq{Email: in.GetUserEmail()})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetLbkUserDetail on ListDepartmentReports", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}
	if len(userInfo.OwnerDepartmentList) == 0 {
		return nil, common.ErrPermissionDenied
	}

	departmentKey, err := getManagedDepartmentKeyList(userInfo, in.GetDepartmentKey())
	if err != nil {
		return nil, err
	}

	getDepartmentDirectlyUserReq := usercenter.GetDepartmentDirectlyUserReq{
		AppKey:                       l.svcCtx.Config.LbkUsercenter.AppName,
		AppToken:                     l.svcCtx.Config.LbkUsercenter.AppToken,
		DepartmentKey:                departmentKey,
		IncludeSecondDepartmentOwner: 1, // 默认不包含二级部门负责人
	}

	// 当rockey查看 研发部的工作汇报时，不包含其子部门的人
	if in.GetUserEmail() == globalkey.RockyEmail && departmentKey == globalkey.RDDepartmentKey {
		getDepartmentDirectlyUserReq.IncludeChildrenDepartment = 1    // 1：不包含；0：包含
		getDepartmentDirectlyUserReq.IncludeSecondDepartmentOwner = 0 // 1：不包含；0：包含
	}
	departmentUsers, err := l.svcCtx.LbkUcClient.GetDepartmentDirectlyUser(l.ctx, getDepartmentDirectlyUserReq)
	if err != nil {
		logx.Errorw("failed to call usercenter.GetDepartmentDirectlyUser on ListDepartmentReports", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}
	if len(departmentUsers) == 0 {
		return &oa.ListDepartmentReportsResp{}, nil
	}

	emailList := getUserEmailFromUsers(departmentUsers)
	q := report.FindDepartmentReportsReq{
		EmailList:    emailList,
		ReportType:   int32(in.GetReportType()),
		ReportDate:   reportDate,
		ReportStatus: int32(in.GetReportStatus()),
		IsDelayed:    in.GetIsDelayed(),
		HasRisk:      in.GetHasRisk(),
	}
	if q.ReportStatus == 0 {
		// 默认查询已提交的工作汇报。
		q.ReportStatus = int32(oa.ReportStatus_DONE_RS)
	}
	reportList, err := l.svcCtx.ReportModel.FindDepartmentReports(l.ctx, q)
	if err != nil {
		logx.Errorw("failed to query x_report on ListDepartmentReports", logx.Field("error", err))
		return nil, ErrReportQueryFailed
	}
	reportItemList, err := l.svcCtx.ReportItemModel.FindByReportIds(l.ctx, reportList.Ids())
	if err != nil {
		logx.Errorw("failed to query x_report_item on ListDepartmentReports", logx.Field("error", err))
		return nil, ErrReportItemQueryFailed
	}

	m := make(map[string]struct{}, len(reportList))
	var list []*oa.Report
	for _, r := range reportList {
		m[r.UserEmail] = struct{}{}
		if in.GetReportStatus() == oa.ReportStatus_DRAFT_RS &&
			r.Status == uint64(oa.ReportStatus_DONE_RS) {
			// 当查询没有交的人时，要区分草稿和没创建汇报的。
			// 此时 需要过滤掉已经提交的
			continue
		}
		itemList := reportItemList.GetByReportId(r.Id)
		items := make([]*oa.ReportItem, len(itemList))
		for j, v := range itemList {
			items[j] = &oa.ReportItem{
				ReportItemId: v.Id,
				Content:      v.Content,
			}
		}

		rpt := &oa.Report{
			ReportId:       r.Id,
			ReportType:     oa.ReportType(r.ReportType),
			ReportDate:     r.ReportDate.Local().Format(time.DateOnly),
			IsDelayed:      convertIn64ToBool(r.IsDelayed),
			DelayReason:    r.DelayReason,
			HasRisk:        convertIn64ToBool(r.HasRisk),
			RiskDesc:       r.RiskDesc,
			ReportStatus:   oa.ReportStatus(r.Status),
			CreatedAt:      r.CreatedAt.Local().Format(time.DateTime),
			UpdatedAt:      r.UpdatedAt.Local().Format(time.DateTime),
			ReportItemList: items,
			UserEmail:      r.UserEmail,
		}

		// 填充用户部门信息
		user := common.GetUserByEmail(departmentUsers, r.UserEmail)
		if user != nil && len(user.SecondDepartmentList) > 0 {
			rpt.Department = &oa.DepartmentInfo{
				Key:  user.SecondDepartmentList[0].DepartmentKey,
				Name: user.SecondDepartmentList[0].DepartmentName,
			}
		}

		list = append(list, rpt)
	}

	// 填充未交工作汇报的人
	// 当前端未选择is_delayed, has_risk条件时(is_delayed=-1,has_risk=-1)，填充未提交汇报的人
	// 当前端未选择report_status(0),或者查询草稿状态(1)时，需要填充未提交汇报的人。
	if in.GetIsDelayed() < 0 && in.GetHasRisk() < 0 && in.GetReportStatus() != oa.ReportStatus_DONE_RS {
		for _, u := range emailList {
			if _, ok := m[u]; !ok {
				rpt := &oa.Report{UserEmail: u}
				// 填充用户部门信息
				user := common.GetUserByEmail(departmentUsers, u)
				if user != nil && len(user.SecondDepartmentList) > 0 {
					rpt.Department = &oa.DepartmentInfo{
						Key:  user.SecondDepartmentList[0].DepartmentKey,
						Name: user.SecondDepartmentList[0].DepartmentName,
					}
				}

				list = append(list, rpt)
			}
		}
	}
	// 按照 department_name 排序
	// 创建一个中文排序器
	c := collate.New(language.Chinese)
	sort.Slice(list, func(i, j int) bool {
		if list[i].Department == nil && list[j].Department == nil {
			return false
		}
		if list[i].Department == nil {
			return false
		}
		if list[j].Department == nil {
			return true
		}
		return c.CompareString(list[i].Department.Name, list[j].Department.Name) == -1
	})

	return &oa.ListDepartmentReportsResp{List: list}, nil
}

func (l *ListDepartmentReportsLogic) Validate(in *oa.ListDepartmentReportsReq) (time.Time, error) {
	reportDate, err := time.Parse(time.DateOnly, in.GetReportDate())
	if err != nil {
		return reportDate, status.Error(codes.InvalidArgument, "report_date参数格式错误")
	}
	if in.GetUserEmail() == "" {
		return reportDate, ErrUserEmailRequired
	}
	if in.GetReportType() == 0 {
		return reportDate, status.Error(codes.InvalidArgument, "report_type参数必填")
	}

	return reportDate, nil
}

func getManagedDepartmentKeyList(userInfo *usercenter.PublicLbkUserOutside, dpKey string) (string, error) {
	var departmentKey string
	switch userInfo.Email {
	case globalkey.RockyEmail:
		// rocky默认展示"研发部"下面的成员。
		departmentKey = globalkey.RDDepartmentKey
		if dpKey != "" {
			//if !common.IsContainDepartment(userInfo.SecondDepartmentList, dpKey) &&
			if !common.IsContainDepartment(userInfo.DepartmentList, dpKey) &&
				!common.IsContainDepartment(userInfo.OwnerDepartmentList, dpKey) {
				return "", common.ErrPermissionDenied
			}
			departmentKey = dpKey
		}
	default:
		if len(userInfo.OwnerDepartmentList) > 0 {
			// 其他人默认展示管理部门中的第一个
			departmentKey = userInfo.OwnerDepartmentList[0].DepartmentKey
			if dpKey != "" {
				//if !common.IsContainDepartment(userInfo.OwnerDepartmentList, dpKey)
				if !common.IsContainDepartment(userInfo.DepartmentList, dpKey) {
					return "", common.ErrPermissionDenied
				}
				departmentKey = dpKey
			}
		}
	}

	if departmentKey == "" {
		return "", common.ErrPermissionDenied
	}

	return departmentKey, nil
}
