package reportrpclogic

import (
	"context"
	"maps"
	"slices"
	"time"

	"oa-server/app/oacenter/model/report"
	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type ListReportDelayAndRiskLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewListReportDelayAndRiskLogic(ctx context.Context, svcCtx *svc.ServiceContext) *ListReportDelayAndRiskLogic {
	return &ListReportDelayAndRiskLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取汇报的延期、风险列表
func (l *ListReportDelayAndRiskLogic) ListReportDelayAndRisk(in *oa.ListReportDelayAndRiskReq) (*oa.ListReportDelayAndRiskResp, error) {
	startDate, endDate, err := l.Validate(in)
	if err != nil {
		return nil, err
	}

	dpUsers, err := l.svcCtx.LbkUcClient.GetDepartmentDirectlyUser(l.ctx, usercenter.GetDepartmentDirectlyUserReq{
		AppKey:        l.svcCtx.Config.LbkUsercenter.AppName,
		AppToken:      l.svcCtx.Config.LbkUsercenter.AppToken,
		DepartmentKey: in.GetDepartmentKey(),
	})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetDepartmentDirectlyUser on ListReportDelayAndRisk", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}
	emails := getUserEmailFromUsers(dpUsers)

	// 查询部门内当天哪些人交了日报
	now := time.Now()
	reportList, err := l.svcCtx.ReportModel.FindReportsByUserAndReportStartDate(l.ctx, now, emails)
	if err != nil {
		logx.Errorw("failed to query x_report on ListReportDelayAndRisk.FindReportsByUserAndReportstartDate", logx.Field("error", err))
		return nil, ErrReportQueryFailed
	}
	reportMap := reportList.ByUserEmail()

	userNotWrite := make(map[string]struct{})
	if startDate.Unix() < now.Unix() && now.Unix() < endDate.Unix() {
		for _, u := range emails {
			if _, ok := reportMap[u]; !ok {
				userNotWrite[u] = struct{}{}
			}
		}
	}

	queryReq := report.FindUserLatestDelayAndRiskReq{
		EmailList:       emails,
		ReportStartDate: startDate,
		ReportEndDate:   endDate,
	}
	delayAndRiskList, err := l.svcCtx.ReportModel.FindUserLatestDelayAndRisk(l.ctx, queryReq)
	if err != nil {
		logx.Errorw("failed to query x_report on ListReportDelayAndRisk.ReportModel.FindUserLatestDelayAndRisk", logx.Field("error", err))
		return nil, ErrReportQueryFailed
	}

	// key: user_email
	m := make(map[string]*oa.ReportDelayAndRisk)
	for _, v := range delayAndRiskList {
		rs, ok := m[v.UserEmail]
		if !ok {
			m[v.UserEmail] = &oa.ReportDelayAndRisk{
				UserEmail: v.UserEmail,
				IsDelayed: convertIn64ToBool(v.IsDelayed),
				HasRisk:   convertIn64ToBool(v.HasRisk),
			}
			continue
		}
		if v.IsDelayed > 0 {
			rs.IsDelayed = convertIn64ToBool(v.IsDelayed)
		}
		if v.HasRisk > 0 {
			rs.HasRisk = convertIn64ToBool(v.HasRisk)
		}
		m[v.UserEmail] = rs
	}

	// 选择其他时间段，不显示未交日报信息
	showLackDaily := now.After(startDate) && now.Before(endDate)
	var list []*oa.ReportDelayAndRisk
	for _, v := range emails {
		isLackDaily := false
		if showLackDaily {
			if _, ok := reportMap[v]; !ok {
				isLackDaily = true
			}
		}

		item := &oa.ReportDelayAndRisk{
			UserEmail:   v,
			IsLackDaily: isLackDaily,
		}

		r, ok := m[v]
		if ok {
			item.IsDelayed = r.IsDelayed
			item.HasRisk = r.HasRisk
		}

		// 日报填写了，并且没有延期、风险时，跳过。
		if !isLackDaily && !ok {
			continue
		}
		list = append(list, item)
	}

	return &oa.ListReportDelayAndRiskResp{
		DelayRiskList:         list,
		ShowInformBtnUserList: slices.Collect(maps.Keys(userNotWrite)),
	}, nil
}

func (l *ListReportDelayAndRiskLogic) Validate(in *oa.ListReportDelayAndRiskReq) (startDate, endDate time.Time, err error) {
	if in.GetDepartmentKey() == "" {
		err = status.Error(codes.InvalidArgument, "department_key参数必填")
		return
	}

	startDate, err = time.ParseInLocation(time.DateOnly, in.GetReportStartDate(), time.Local)
	if err != nil {
		err = status.Error(codes.InvalidArgument, "report_start_date参数格式错误")
		return
	}
	endDate, err = time.ParseInLocation(time.DateOnly, in.GetReportEndDate(), time.Local)
	if err != nil {
		err = status.Error(codes.InvalidArgument, "report_end_date参数格式错误")
		return
	}
	if endDate.Before(startDate) {
		err = status.Error(codes.InvalidArgument, "report_end_date在report_start_date之前")
		return
	}

	return
}

func getUserEmailFromUsers(users []*usercenter.PublicLbkUserOutside) []string {
	emails := make([]string, len(users))
	for i, u := range users {
		emails[i] = u.Email
	}
	return emails
}
