package reportrpclogic

import (
	"context"
	"time"

	"oa-server/app/oacenter/model/report"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type ListReportMentionLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewListReportMentionLogic(ctx context.Context, svcCtx *svc.ServiceContext) *ListReportMentionLogic {
	return &ListReportMentionLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

// 获取汇报提及列表
func (l *ListReportMentionLogic) ListReportMention(in *oa.ListReportMentionReq) (*oa.ListReportMentionResp, error) {
	startDate, endDate, err := l.Validate(in)
	if err != nil {
		return nil, err
	}

	mentionList, err := l.svcCtx.ReportMentionModel.FindUserReportMention(l.ctx, report.FindUserReportMentionReq{
		UserEmail:       in.GetUserEmail(),
		TargetUserEmail: in.GetTargetUserEmail(),
		ReportStartDate: startDate,
		ReportEndDate:   endDate,
	})
	if err != nil {
		logx.Errorw("failed to query x_report_mention on ListReportMention.ReportMentionModel.FindUserReportMention", logx.Field("error", err))
		return nil, ErrReportMentionQueryFailed
	}
	if len(mentionList) == 0 {
		return &oa.ListReportMentionResp{}, nil
	}
	itemIds := mentionList.ReportItemIds()

	reportItemList, err := l.svcCtx.ReportItemModel.FindByIds(l.ctx, itemIds)
	if err != nil {
		logx.Errorw("failed to query x_report_item on ListReportMention.ReportItemModel.FindByIds", logx.Field("error", err))
		return nil, ErrReportItemQueryFailed
	}

	var list []*oa.ReportMention
	mentionMap := mentionList.ByReportItemId()
	for itemId, l := range mentionMap {
		reportItem := reportItemList.GetById(itemId)
		if reportItem != nil && len(l) > 0 {
			emails := make([]string, len(l))
			for j, m := range l {
				emails[j] = m.MentionUserEmail
			}
			list = append(list, &oa.ReportMention{
				ReportItemId:       itemId,
				ReportType:         oa.ReportType(l[0].ReportType),
				ReportDate:         l[0].ReportDate.Local().Format(time.DateOnly),
				InitiatorEmail:     l[0].InitiatorEmail,
				MentionItemContent: reportItem.Content,
				MentionUserList:    emails,
			})
		}
	}

	return &oa.ListReportMentionResp{Total: int64(len(list)), List: list}, nil
}

func (l *ListReportMentionLogic) Validate(in *oa.ListReportMentionReq) (startDate, endDate time.Time, err error) {
	if in.GetUserEmail() == "" {
		err = ErrUserEmailRequired
		return
	}
	if in.GetTargetUserEmail() == "" {
		err = status.Error(codes.InvalidArgument, "target_user_email参数必填")
		return
	}

	startDate, err = time.ParseInLocation(time.DateOnly, in.GetReportStartDate(), time.Local)
	if err != nil {
		err = status.Error(codes.InvalidArgument, "report_start_date参数格式错误")
		return
	}
	endDate, err = time.ParseInLocation(time.DateOnly, in.GetReportEndDate(), time.Local)
	if err != nil {
		err = status.Error(codes.InvalidArgument, "report_end_date参数格式错误")
		return
	}
	if endDate.Before(startDate) {
		err = status.Error(codes.InvalidArgument, "report_end_date在report_start_date之前")
		return
	}

	return
}
