package userrpclogic

import (
	"context"
	"sort"

	"oa-server/app/oacenter/oa_rpc/internal/logic/common"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	"github.com/zeromicro/go-zero/core/logx"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

type GetDepartmentUserListLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

func NewGetDepartmentUserListLogic(ctx context.Context, svcCtx *svc.ServiceContext) *GetDepartmentUserListLogic {
	return &GetDepartmentUserListLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *GetDepartmentUserListLogic) GetDepartmentUserList(in *oa.GetDepartmentUserListReq) (*oa.GetDepartmentUserListResp, error) {
	if len(in.GetDepartmentKeyList()) == 0 {
		return nil, status.Error(codes.InvalidArgument, "department_key_list参数必填")
	}

	dpUsers, err := l.svcCtx.LbkUcClient.GetDepartmentDirectlyUser(l.ctx, usercenter.GetDepartmentDirectlyUserReq{
		AppKey:           l.svcCtx.Config.LbkUsercenter.AppName,
		AppToken:         l.svcCtx.Config.LbkUsercenter.AppToken,
		DeparmentKeyList: in.GetDepartmentKeyList(),
	})
	if err != nil {
		logx.Errorw("failed to call usercenter.GetDepartmentUser On GetDepartmentUserList", logx.Field("error", err))
		return nil, common.ErrUsercenterUnavailable
	}

	// key:department_key, value: department_name
	targetDepartmentMap := make(map[string]string)
	for _, v := range in.GetDepartmentKeyList() {
		targetDepartmentMap[v] = ""
	}

	dpUsersMap := make(map[string][]*oa.UserInfo)
	dpLeaders := make(map[string][]string)
	for _, u := range dpUsers {
		for _, d := range u.SecondDepartmentList {
			if _, ok := targetDepartmentMap[d.DepartmentKey]; ok {
				// 记录部门名称
				targetDepartmentMap[d.DepartmentKey] = d.DepartmentName
				u2 := &oa.UserInfo{
					Nickname:        u.NickName,
					Email:           u.Email,
					LarkHeaderImage: u.LarkHeaderImg,
					UserStatus:      int32(u.UserStatus),
					IsLeader:        false,
					JoinTime:        u.JoinTime,
				}

				// 收集部门负责人，包括二级、三级部门
				if len(u.OwnerDepartmentList) > 0 {
					u2.IsLeader = true
					dpLeaders[d.DepartmentKey] = append(dpLeaders[d.DepartmentKey], u.Email)
				}
				dpUsersMap[d.DepartmentKey] = append(dpUsersMap[d.DepartmentKey], u2)
				break
			}
		}
	}

	var dpUserList []*oa.DepartmentUser
	for _, k := range in.GetDepartmentKeyList() {
		userList, ok := dpUsersMap[k]
		if !ok {
			continue
		}

		// 对userList排序
		sort.SliceStable(userList, func(i, j int) bool {
			if userList[i].IsLeader && !userList[j].IsLeader {
				return true
			}
			if !userList[i].IsLeader && userList[j].IsLeader {
				return false
			}
			return i < j
		})

		dpUserList = append(dpUserList, &oa.DepartmentUser{
			DepartmentKey:     k,
			DepartmentName:    targetDepartmentMap[k],
			UserList:          userList,
			DepartmentLeaders: dpLeaders[k],
		})
	}

	return &oa.GetDepartmentUserListResp{DepartmentUserList: dpUserList}, nil
}
