package userrpclogic

import (
	"context"
	"encoding/json"
	"net/http"
	"net/url"
	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"oa-server/common/httputil"
	"time"

	"github.com/zeromicro/go-zero/core/logx"
)

type UserListLogic struct {
	ctx    context.Context
	svcCtx *svc.ServiceContext
	logx.Logger
}

// DepartmentUserReq 用户中心部门用户请求
type DepartmentUserReq struct {
	DepartmentKeyList []string `json:"department_key_list"`
}

// DepartmentUserResp 用户中心部门用户响应
type DepartmentUserResp struct {
	Code    int              `json:"code"`
	Message string           `json:"message"`
	Data    []model.UserInfo `json:"data"`
}

func NewUserListLogic(ctx context.Context, svcCtx *svc.ServiceContext) *UserListLogic {
	return &UserListLogic{
		ctx:    ctx,
		svcCtx: svcCtx,
		Logger: logx.WithContext(ctx),
	}
}

func (l *UserListLogic) UserList(in *oa.UserListReq) (*oa.UserListResp, error) {
	users := []model.UserInfo{}
	var err error
	if len(in.DepartmentKeys) == 0 {
		cache := model.NewUserCenterService(l.svcCtx.RedisClient)
		users, err = cache.GetAllUsers(l.ctx)
		if err != nil {
			logx.Error("get user list error", err)
			return nil, err
		}
	} else {
		// 从用户中心获取指定部门用户
		reqBody := DepartmentUserReq{
			DepartmentKeyList: in.DepartmentKeys,
		}
		jsonData, err := json.Marshal(reqBody)
		if err != nil {
			logx.Error("marshal request error", err)
			return nil, err
		}
		client := httputil.NewClient(30*time.Second, false)
		userUrl, err := url.JoinPath(l.svcCtx.Config.UserCenter.URL, l.svcCtx.Config.UserCenter.DepartmentUserPath)
		if err != nil {
			logx.Error("join path error", err)
			return nil, err
		}
		resp := DepartmentUserResp{}
		_, _, _, err = client.Post(l.ctx, userUrl, http.Header{
			"Content-Type": []string{"application/json"},
		}, jsonData, &resp)

		if err != nil {
			logx.Error("do request error", err)
			return nil, err
		}

		users = resp.Data
	}

	userList := make([]*oa.UserInfo, len(users))
	for i, v := range users {
		userList[i] = &oa.UserInfo{
			Id:              v.ID,
			Nickname:        v.NickName,
			Email:           v.Email,
			LarkHeaderImage: v.LarkHeaderImg,
			Departments:     []*oa.DepartmentInfo{},
		}
	}
	return &oa.UserListResp{
		List:  userList,
		Total: int64(len(users)),
	}, nil
}
