package svc

import (
	"log"
	"oa-server/app/jobcenter/model"
	"oa-server/app/oacenter/model/efficiency_calc"
	"oa-server/app/oacenter/model/issue"
	"oa-server/app/oacenter/model/monitor_metrics"
	"oa-server/app/oacenter/model/okr"
	"oa-server/app/oacenter/model/performance"
	"oa-server/app/oacenter/model/report"
	"oa-server/app/oacenter/oa_rpc/internal/config"
	"oa-server/common/msgcenter"

	"git.lbk.world/test/devops/lbk-go-sdk/usercenter"
	redisV9 "github.com/redis/go-redis/v9"
	"github.com/zeromicro/go-zero/core/stores/cache"
	"github.com/zeromicro/go-zero/core/stores/redis"
	"github.com/zeromicro/go-zero/core/stores/sqlx"
)

type ServiceContext struct {
	Config                        config.Config
	MonitorMetricsModel           monitor_metrics.XMonitorMetricsModel
	RedisClient                   *redisV9.Client
	IssueModel                    issue.XIssueMetricsModel
	PerformanceModel              performance.XPerformanceModel
	PerformanceWorkModel          performance.XPerformanceWorkModel
	PerformanceAddtionModel       performance.XPerformanceAdditionModel
	PerformanceItemCommentModel   performance.XPerformanceItemCommentModel
	ReportModel                   report.XReportModel
	ReportItemModel               report.XReportItemModel
	ReportCommentModel            report.XReportCommentModel
	ReportMentionModel            report.XReportMentionModel
	EfficiencyCalcModel           efficiency_calc.XEfficiencyCalcModel
	EfficiencyCalcHistoryModel    efficiency_calc.XEfficiencyCalcHistoryModel
	OkrPeriodModel                okr.XOkrPeriodModel
	OkrTaskModel                  okr.XOKrTaskModel
	OkrAlignmentModel             okr.XOkrAlignmentModel
	OkrMsgModel                   okr.XOkrMsgModel
	OkrHistoryModel               okr.XOkrOperatingHistoryModel
	LarkService                   *msgcenter.LarkService
	LbkUcClient                   *usercenter.LbkUserClient
	UserCache                     *model.UserCenterService
	PerformanceAdditionScoreModel performance.XPerformanceAdditionScoreModel
}

func NewServiceContext(c config.Config) *ServiceContext {
	// 用户中心客户端
	lbkUcClient, _ := usercenter.NewLbkUserClient(c.LbkUsercenter.Env, c.LbkUsercenter.HostInAli)

	// lark
	larkService, err := msgcenter.NewLarkService(msgcenter.LarkConfig{
		AppID:     c.Lark.AppId,
		AppSecret: c.Lark.AppSecret,
		Env:       c.Env,
	})
	if err != nil {
		log.Fatal("failed to create msgcenter.LarkService ", err)
	}

	// 创建缓存配置
	var cacheConf = []cache.NodeConf{
		{
			RedisConf: redis.RedisConf{
				Host: c.RedisConf.Host,
				Type: c.RedisConf.Type,
				Pass: c.RedisConf.Pass,
			},
			Weight: 100,
		},
	}
	svcCtx := &ServiceContext{
		Config:              c,
		MonitorMetricsModel: monitor_metrics.NewXMonitorMetricsModel(sqlx.NewMysql(c.DB.DataSource), cacheConf),
		RedisClient: redisV9.NewClient(&redisV9.Options{
			Addr:     c.RedisConf.Host,
			Password: c.RedisConf.Pass,
		}),
		IssueModel:                    issue.NewXIssueMetricsModel(sqlx.NewMysql(c.DB.DataSource), cacheConf),
		PerformanceModel:              performance.NewXPerformanceModel(sqlx.NewMysql(c.DB.DataSource)),
		PerformanceWorkModel:          performance.NewXPerformanceWorkModel(sqlx.NewMysql(c.DB.DataSource)),
		PerformanceAddtionModel:       performance.NewXPerformanceAdditionModel(sqlx.NewMysql(c.DB.DataSource)),
		PerformanceItemCommentModel:   performance.NewXPerformanceItemCommentModel(sqlx.NewMysql(c.DB.DataSource)),
		PerformanceAdditionScoreModel: performance.NewXPerformanceAdditionScoreModel(sqlx.NewMysql(c.DB.DataSource)),
		ReportModel:                   report.NewXReportModel(sqlx.NewMysql(c.DB.DataSource)),
		ReportItemModel:               report.NewXReportItemModel(sqlx.NewMysql(c.DB.DataSource)),
		ReportCommentModel:            report.NewXReportCommentModel(sqlx.NewMysql(c.DB.DataSource)),
		ReportMentionModel:            report.NewXReportMentionModel(sqlx.NewMysql(c.DB.DataSource)),
		EfficiencyCalcModel:           efficiency_calc.NewXEfficiencyCalcModel(sqlx.NewMysql(c.DB.DataSource), cacheConf),
		EfficiencyCalcHistoryModel:    efficiency_calc.NewXEfficiencyCalcHistoryModel(sqlx.NewMysql(c.DB.DataSource), cacheConf),
		OkrPeriodModel:                okr.NewXOkrPeriodModel(sqlx.NewMysql(c.DB.DataSource)),
		OkrTaskModel:                  okr.NewXOKrTaskModel(sqlx.NewMysql(c.DB.DataSource)),
		OkrAlignmentModel:             okr.NewXOkrAlignmentModel(sqlx.NewMysql(c.DB.DataSource)),
		OkrMsgModel:                   okr.NewXOkrMsgModel(sqlx.NewMysql(c.DB.DataSource)),
		OkrHistoryModel:               okr.NewXOkrOperatingHistoryModel(sqlx.NewMysql(c.DB.DataSource)),
		LarkService:                   larkService,
		LbkUcClient:                   lbkUcClient,
	}
	svcCtx.UserCache = model.NewUserCenterService(svcCtx.RedisClient)
	return svcCtx
}
