package main

import (
	"flag"
	"fmt"
	"log"
	"oa-server/app/oacenter/oa_rpc/internal/config"
	"oa-server/app/oacenter/oa_rpc/internal/cron"
	calcrpcServer "oa-server/app/oacenter/oa_rpc/internal/server/calcrpc"
	issuerpcServer "oa-server/app/oacenter/oa_rpc/internal/server/issuerpc"
	monitorrpcServer "oa-server/app/oacenter/oa_rpc/internal/server/monitorrpc"
	okrrpcServer "oa-server/app/oacenter/oa_rpc/internal/server/okrrpc"
	performanceRpcServer "oa-server/app/oacenter/oa_rpc/internal/server/performancerpc"
	reportRpcServer "oa-server/app/oacenter/oa_rpc/internal/server/reportrpc"
	userrpcServer "oa-server/app/oacenter/oa_rpc/internal/server/userrpc"
	"oa-server/app/oacenter/oa_rpc/internal/svc"
	"oa-server/app/oacenter/oa_rpc/oa"
	"os"
	"strings"

	"github.com/golang-migrate/migrate/v4"
	_ "github.com/golang-migrate/migrate/v4/database/mysql"
	_ "github.com/golang-migrate/migrate/v4/source/file"

	"github.com/zeromicro/go-zero/core/conf"
	"github.com/zeromicro/go-zero/core/service"
	"github.com/zeromicro/go-zero/zrpc"
	"google.golang.org/grpc"
	"google.golang.org/grpc/reflection"
)

var configFile = flag.String("f", "etc/oarpc.yaml", "the config file")

func main() {
	flag.Parse()

	var c config.Config
	conf.MustLoad(*configFile, &c)
	ctx := svc.NewServiceContext(c)

	s := zrpc.MustNewServer(c.RpcServerConf, func(grpcServer *grpc.Server) {
		oa.RegisterUserRpcServer(grpcServer, userrpcServer.NewUserRpcServer(ctx))
		oa.RegisterMonitorRpcServer(grpcServer, monitorrpcServer.NewMonitorRpcServer(ctx))
		oa.RegisterPerformanceRpcServer(grpcServer, performanceRpcServer.NewPerformanceRpcServer(ctx))
		oa.RegisterReportRpcServer(grpcServer, reportRpcServer.NewReportRpcServer(ctx))
		oa.RegisterIssueRpcServer(grpcServer, issuerpcServer.NewIssueRpcServer(ctx))
		oa.RegisterCalcRpcServer(grpcServer, calcrpcServer.NewCalcRpcServer(ctx))
		oa.RegisterOkrRpcServer(grpcServer, okrrpcServer.NewOkrRpcServer(ctx))
		if c.Mode == service.DevMode || c.Mode == service.TestMode {
			reflection.Register(grpcServer)
		}
	})
	defer s.Stop()
	MigrateSql(c)
	cron.Init(ctx)
	fmt.Printf("Starting rpc server at %s...\n", c.ListenOn)
	s.Start()
}

type customLogger struct {
	*log.Logger
	verbose bool
}

func (l *customLogger) Verbose() bool {
	return true // 设置为 true 将记录所有日志
}
func MigrateSql(c config.Config) {
	if c.Env == "local" {
		return
	}
	// 创建详细日志记录器
	logger := &customLogger{
		Logger: log.New(os.Stdout, "MIGRATE: ", log.Ldate|log.Ltime),
	}
	urls := strings.Split(c.DB.DataSource, "?")
	// 数据库连接字符串
	dsn := fmt.Sprintf("mysql://%s?multiStatements=true", urls[0])

	// 设置详细日志
	sourceURL := "file://" + "./app/oacenter/model/sql"

	m, err := migrate.New(sourceURL, dsn)
	if err != nil {
		log.Printf("连接数据库失败: %v \n", err)
		return
	}

	m.Log = logger

	if err := m.Up(); err != nil && err != migrate.ErrNoChange {
		log.Printf("数据库迁移失败: %v \n", err)
		return
	}

	log.Println("数据库迁移成功执行")
}
