package httputil

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net"
	"net/http"
	"strings"
	"time"

	"github.com/zeromicro/go-zero/core/logx"
)

type Client struct {
	client *http.Client
}

func NewClient(timeout time.Duration, disableKeepAlives bool) *Client {
	return &Client{
		client: &http.Client{
			Timeout: timeout,
			Transport: &http.Transport{
				Proxy: http.ProxyFromEnvironment,
				DialContext: (&net.Dialer{
					Timeout:   30 * time.Second,
					KeepAlive: 30 * time.Second,
				}).DialContext,
				ForceAttemptHTTP2:     true,
				MaxIdleConns:          3,
				IdleConnTimeout:       10 * time.Second,
				TLSHandshakeTimeout:   10 * time.Second,
				ExpectContinueTimeout: 1 * time.Second,
				DisableKeepAlives:     disableKeepAlives,
			},
		},
	}
}

// Request 发送HTTP请求
// method: HTTP方法 (GET, POST, PUT, DELETE等)
// url: 请求URL
// headers: 请求头
// body: 请求体
// value: 响应数据的结构体指针
// 返回: 响应对象、响应体、是否发送请求、错误信息
func (c *Client) Request(ctx context.Context, method, url string, headers http.Header, body []byte, value interface{}) (resp *http.Response, responseBody []byte, requestSent bool, err error) {
	method = strings.ToUpper(method)
	start := time.Now()

	defer func() {
		duration := time.Since(start)
		var respStatusCode int
		var respHeaders http.Header
		if resp != nil {
			respStatusCode = resp.StatusCode
			respHeaders = resp.Header
		}
		if err != nil {
			logx.Errorf("HTTP请求失败: %+v, 方法: %s, URL: %s, 请求头: %+v, 请求体: %s, 开始时间: %v, 耗时: %v, 响应状态码: %d, 响应头: %+v, 响应体: %s",
				err, method, url, headers, string(body), start, duration, respStatusCode, respHeaders, string(responseBody))
		}
	}()

	// 创建请求
	req, err := http.NewRequestWithContext(ctx, method, url, bytes.NewReader(body))
	if err != nil {
		return nil, nil, false, fmt.Errorf("创建请求失败: %w", err)
	}

	// 设置请求头
	req.Header = headers
	if method == "POST" || method == "PUT" {
		if req.Header.Get("Content-Type") == "" {
			req.Header.Set("Content-Type", "application/json")
		}
	}

	// 发送请求
	requestSent = true
	resp, err = c.client.Do(req)
	if err != nil {
		return nil, nil, true, fmt.Errorf("发送请求失败: %w", err)
	}
	defer func() {
		if closeErr := resp.Body.Close(); closeErr != nil {
			logx.Errorf("关闭响应体失败: %+v", closeErr)
		}
	}()

	// 读取响应体
	responseBody, err = io.ReadAll(resp.Body)
	if err != nil {
		return resp, nil, true, fmt.Errorf("读取响应体失败: %w", err)
	}

	// 解析响应数据
	if value != nil {
		if err := json.Unmarshal(responseBody, value); err != nil {
			return resp, responseBody, true, fmt.Errorf("解析响应数据失败: %w", err)
		}
	}

	return resp, responseBody, true, nil
}

// Get 发送GET请求
func (c *Client) Get(ctx context.Context, url string, headers http.Header, value interface{}) (*http.Response, []byte, bool, error) {
	return c.Request(ctx, "GET", url, headers, nil, value)
}

// Post 发送POST请求
func (c *Client) Post(ctx context.Context, url string, headers http.Header, body []byte, value interface{}) (*http.Response, []byte, bool, error) {
	return c.Request(ctx, "POST", url, headers, body, value)
}

// Put 发送PUT请求
func (c *Client) Put(ctx context.Context, url string, headers http.Header, body []byte, value interface{}) (*http.Response, []byte, bool, error) {
	return c.Request(ctx, "PUT", url, headers, body, value)
}

// Delete 发送DELETE请求
func (c *Client) Delete(ctx context.Context, url string, headers http.Header, value interface{}) (*http.Response, []byte, bool, error) {
	return c.Request(ctx, "DELETE", url, headers, nil, value)
}
