package msgcenter

import (
	"bytes"
	"context"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	lark "github.com/larksuite/oapi-sdk-go/v3"
	larkim "github.com/larksuite/oapi-sdk-go/v3/service/im/v1"
	"github.com/zeromicro/go-zero/core/logx"
	"io/ioutil"
	"net/http"
)

type LarkConfig struct {
	AppID     string
	AppSecret string
	Env       string
}

type LarkService struct {
	Client *lark.Client
}

func NewLarkService(cfg LarkConfig) (*LarkService, error) {
	if cfg.Env == "develop" { // 修改此处一定确认只推送到本人或者测试群
		return nil, errors.New("开发环境不发送飞书消息")
	}

	client := lark.NewClient(cfg.AppID, cfg.AppSecret, lark.WithOpenBaseUrl(lark.LarkBaseUrl))
	return &LarkService{Client: client}, nil
}

// BuildCardContent
//
//	@function:		BuildCardContent
//	@description:	构建卡片内容
//	@params			templateID string 卡片模板ID，在飞书的消息卡片搭建工具中获取：https://open.larksuite.com/tool/cardbuilder?from=howtoguide&templateId=ctp_AAr5UIOlblgm
//	@params			templateVars map[string]interface{} 卡片中自定义的变量
//	@return:		string error
func (ls *LarkService) BuildCardContent(templateID string, templateVars map[string]interface{}) (string, error) {
	cardContent := &NewCardContent{
		Type: "template",
		Data: &NewCardContentData{
			TemplateID:   templateID,
			TemplateVars: templateVars,
		},
	}
	content, err := cardContent.String()
	logx.Errorf("BuildCardContentContent:" + content)
	if err != nil {
		logx.Errorf("BuildCardContentError:" + content + err.Error())
		return "", nil
	}
	return content, nil
}

// SendCardMessage
// 创建lark client并发送消息卡片
//
//	@function:		SendCardMessage
//	@description:	发送飞书卡片信息
//	@params			receiveIdType string 发送的卡片类型e.g. ReceiveIdTypeEmail是单独发消息，ReceiveIdTypeChatId发送到群里，并通过email@人
//	@params			receiveId string 发送给指定人,公司里边直接用email就可以,如果是群类型，则是需要获取该群的机器人 chat_id，通过以下链接获取
//
// https://open.larksuite.com/api-explorer/cli_a4c8d4bf2af81009?apiName=list&from=op_develop_app&project=im&resource=chat&version=v1
//
//	@params			msgType string 发送的消息类型 如MsgTypeInteractive 卡片类型
//	@params			content string发送的文字内容
//	@return:		err error
func (ls *LarkService) SendCardMessage(receiveIdType string, receiveId string, msgType string, content string) error {
	resp, err := ls.Client.Im.Message.Create(context.Background(),
		larkim.NewCreateMessageReqBuilder().ReceiveIdType(receiveIdType).
			Body(larkim.NewCreateMessageReqBodyBuilder().MsgType(msgType).
				ReceiveId(receiveId).
				Content(content).Build()).Build())
	if !resp.Success() || err != nil {
		logx.Errorf(fmt.Sprintf("lark发送消息失败:code = %d msg=%s RequestId=%s receiveId=%s, content=%s",
			resp.Code, resp.Msg, resp.RequestId(), receiveId, content))
		return errors.New(resp.Msg)
	}
	return nil
}

func SendMessage(msg interface{}, hookUrl string) error {
	client := &http.Client{}
	body := bytes.NewBuffer(nil)
	err := json.NewEncoder(body).Encode(msg)
	if err != nil {
		return err
	}
	req, err := http.NewRequest(http.MethodPost, hookUrl, body)
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")
	res, err := client.Do(req)
	if err != nil {
		return err
	}
	defer res.Body.Close()

	result, err := ioutil.ReadAll(res.Body)
	if res.StatusCode != 200 {
		return fmt.Errorf("send dingTalk message failed, %s", httpError(req, res, result, "http code is not 200"))
	}
	if err != nil {
		return fmt.Errorf("send dingTalk message failed, %s", httpError(req, res, result, err.Error()))
	}

	type response struct {
		ErrCode int `json:"errcode"`
	}
	var ret response

	if err := json.Unmarshal(result, &ret); err != nil {
		return fmt.Errorf("send dingTalk message failed, %s", httpError(req, res, result, err.Error()))
	}

	if ret.ErrCode != 0 {
		return fmt.Errorf("send dingTalk message failed, %s", httpError(req, res, result, "errcode is not 0"))
	}

	return nil
}

func httpError(request *http.Request, response *http.Response, body []byte, error string) string {
	return fmt.Sprintf(
		"http request failure, error: %s, status code: %d, %s %s, body:\n%s",
		error,
		response.StatusCode,
		request.Method,
		request.URL.String(),
		string(body),
	)
}

func GenSign(secret string, timestamp int64) (string, error) {
	// timestamp + key 做sha256, 再进行base64 encode
	stringToSign := fmt.Sprintf("%v", timestamp) + "\n" + secret
	var data []byte
	h := hmac.New(sha256.New, []byte(stringToSign))
	_, err := h.Write(data)
	if err != nil {
		return "", err
	}
	signature := base64.StdEncoding.EncodeToString(h.Sum(nil))
	return signature, nil
}

func (ls *LarkService) BuildContentAndSendLarkMsg(tplID, chatID string, vars map[string]interface{}) error {
	// tplID 模版ID
	// chatID EP机器人所在的群ID
	// vars 模版变量
	content, err := ls.BuildCardContent(tplID, vars)
	if err != nil {
		logx.Errorf("BuildCardContentErr:" + content + err.Error())
		return err
	}
	err = ls.SendCardMessage(larkim.ReceiveIdTypeChatId, chatID, larkim.MsgTypeInteractive, content)
	if err != nil {
		logx.Errorf("BuildCardContentErr:" + content + err.Error())
		return err
	}
	return nil
}
