package period

import (
	"fmt"
	"strings"
	"time"
)

// 周期方向枚举
type PeriodDirection int

const (
	PeriodDirectionForward  PeriodDirection = 1 // 向前获取
	PeriodDirectionBackward PeriodDirection = 2 // 向后获取
)

// GetPeriodList 获取周期列表
// periodType: 周期类型(1:周, 2:月, 3:双月, 4:季度)
// dateString: 日期范围字符串，格式如"20250101-20250131"
// count: 需要返回的周期数量(包含当前周期)
// direction: 获取方向(1:向前, 2:向后)
func GetPeriodList(periodType int, dateString string, count int, direction PeriodDirection) ([]string, error) {
	// 解析日期范围
	dates := strings.Split(dateString, "-")
	if len(dates) != 2 {
		return nil, fmt.Errorf("invalid date format: %s", dateString)
	}

	// 解析开始日期和结束日期
	startDate, err := time.Parse("20060102", dates[0])
	if err != nil {
		return nil, fmt.Errorf("invalid start date: %v", err)
	}

	endDate, err := time.Parse("20060102", dates[1])
	if err != nil {
		return nil, fmt.Errorf("invalid end date: %v", err)
	}

	periods := make([]string, 0, count)
	currentStartDate := startDate
	currentEndDate := endDate

	// 添加第一个周期（即当前周期）
	periods = append(periods, fmt.Sprintf("%s-%s", currentStartDate.Format("20060102"), currentEndDate.Format("20060102")))

	// 遍历生成剩余周期
	for i := 1; i < count; i++ {
		var nextStartDate, nextEndDate time.Time

		switch periodType {
		case 1: // 周
			if direction == PeriodDirectionForward {
				// 向前移动一周
				nextEndDate = currentStartDate.AddDate(0, 0, -1)
				nextStartDate = nextEndDate.AddDate(0, 0, -6)
			} else {
				// 向后移动一周
				nextStartDate = currentEndDate.AddDate(0, 0, 1)
				nextEndDate = nextStartDate.AddDate(0, 0, 6)
			}
		case 2: // 月
			if direction == PeriodDirectionForward {
				// 向前移动一个月
				// 当前开始日期的前一天的月初
				prevMonth := currentStartDate.AddDate(0, -1, 0)
				nextStartDate = time.Date(prevMonth.Year(), prevMonth.Month(), 1, 0, 0, 0, 0, time.Local)
				// 当前开始日期的前一天
				nextEndDate = currentStartDate.AddDate(0, 0, -1)
			} else {
				// 向后移动一个月
				// 当前结束日期的后一天
				nextStartDate = currentEndDate.AddDate(0, 0, 1)
				// 下个月月末
				nextMonth := nextStartDate.AddDate(0, 1, 0)
				nextEndDate = time.Date(nextMonth.Year(), nextMonth.Month(), 0, 0, 0, 0, 0, time.Local)
			}
		case 3: // 双月
			if direction == PeriodDirectionForward {
				// 向前移动两个月
				// 获取当前开始日期的前两个月的月初
				prevTwoMonths := currentStartDate.AddDate(0, -2, 0)
				nextStartDate = time.Date(prevTwoMonths.Year(), prevTwoMonths.Month(), 1, 0, 0, 0, 0, time.Local)
				// 当前开始日期的前一天
				nextEndDate = currentStartDate.AddDate(0, 0, -1)
			} else {
				// 向后移动两个月
				// 当前结束日期的后一天
				nextStartDate = currentEndDate.AddDate(0, 0, 1)
				// 后两个月的月末
				nextTwoMonths := nextStartDate.AddDate(0, 2, 0)
				nextEndDate = time.Date(nextTwoMonths.Year(), nextTwoMonths.Month(), 0, 0, 0, 0, 0, time.Local)
			}
		case 4: // 季度
			if direction == PeriodDirectionForward {
				// 向前移动一个季度（三个月）
				// 获取当前开始日期的前三个月的月初
				prevQuarter := currentStartDate.AddDate(0, -3, 0)
				nextStartDate = time.Date(prevQuarter.Year(), prevQuarter.Month(), 1, 0, 0, 0, 0, time.Local)
				// 当前开始日期的前一天
				nextEndDate = currentStartDate.AddDate(0, 0, -1)
			} else {
				// 向后移动一个季度（三个月）
				// 当前结束日期的后一天
				nextStartDate = currentEndDate.AddDate(0, 0, 1)
				// 后三个月的月末
				nextQuarter := nextStartDate.AddDate(0, 3, 0)
				nextEndDate = time.Date(nextQuarter.Year(), nextQuarter.Month(), 0, 0, 0, 0, 0, time.Local)
			}
		}

		// 将结果添加到周期列表中
		periods = append(periods, fmt.Sprintf("%s-%s", nextStartDate.Format("20060102"), nextEndDate.Format("20060102")))

		// 更新当前日期为下一个周期的日期，为下一次迭代做准备
		currentStartDate = nextStartDate
		currentEndDate = nextEndDate
	}

	return periods, nil
}
