package tool

import (
	"crypto/md5"
	"crypto/rand"
	"encoding/json"
	"fmt"
	"io"
	"mime/multipart"
	"net/http"
	"path/filepath"
	"strconv"
	"sync"
	"time"

	"github.com/aliyun/aliyun-oss-go-sdk/oss"
	"golang.org/x/sync/singleflight"
)

const (
	credentailKey = "oss-sts-key"
	tokenPath     = "/admin-tech-center/tech/oss/generate"
)

var (
	group singleflight.Group
	cache sync.Map
)

type FileService struct {
	config OssConfig
}

type OssConfig struct {
	OssEndpoint   string
	OssBucketName string
	LbkOssApiHost string
	ResourceHost  string
	ObjectPrefix  string
}

func NewFileService(config OssConfig) *FileService {
	return &FileService{config: config}
}

func (fs *FileService) UploadFile(file *multipart.FileHeader) (string, error) {
	// 获取文件类型
	ext := filepath.Ext(file.Filename)

	// 生成随机字符串
	b := make([]byte, 16)
	_, err := rand.Read(b)
	if err != nil {
		return "", err
	}

	// 生成文件名
	now := time.Now()
	rawFileName := fmt.Sprintf("%x-%d", b, now.Unix())
	hash := md5.Sum([]byte(rawFileName))
	newFileName := fmt.Sprintf("%s/%x%s", fs.config.ObjectPrefix, hash, ext)

	// 打开文件
	f, err := file.Open()
	if err != nil {
		return "", err
	}
	defer f.Close()

	ossKeys, err := fs.GetToken()
	// 创建存储空间（Create Bucket）
	client, err := oss.New(fs.config.OssEndpoint, ossKeys.accessKeyID, ossKeys.accessKeySecret, oss.SecurityToken(ossKeys.securityToken))
	if err != nil {
		return "", err
	}

	// 获取存储空间
	bucket, err := client.Bucket(fs.config.OssBucketName)
	if err != nil {
		return "", err
	}

	// 上传文件
	err = bucket.PutObject(newFileName, f)
	//err = bucket.PutObjectFromFile(newFileName, f)
	if err != nil {
		return "", err
	}

	return fs.config.ResourceHost + "/" + newFileName, nil
}

func (fs *FileService) DownloadFile(filepath, targetFilePath string) error {
	ossKeys, err := fs.GetToken()
	if err != nil {
		return err
	}
	// 创建存储空间（Create Bucket）
	client, err := oss.New(fs.config.OssEndpoint, ossKeys.accessKeyID, ossKeys.accessKeySecret, oss.SecurityToken(ossKeys.securityToken))
	if err != nil {
		return err
	}
	bucket, err := client.Bucket(fs.config.OssBucketName)
	if err != nil {
		return err
	}

	// 填写Object的完整路径，完整路径中不能包含Bucket名称，例如exampledir/exampleobject.txt。
	//objectName := "/luban/95fd8cfc227d46ba879018dfb7e0115f.png"

	err = bucket.GetObjectToFile(filepath, targetFilePath)
	if err != nil {
		return err
	}
	return nil
}

func (fs *FileService) DeleteFile() error {
	client, err := oss.New("", "", "")
	if err != nil {
		return err
	}

	bucket, err := client.Bucket("my-bucket")
	if err != nil {
		return err
	}

	err = bucket.DeleteObject("my-object")
	if err != nil {
		return err
	}
	return nil
}

type ossCredentials struct {
	accessKeyID     string
	accessKeySecret string
	securityToken   string
	deadline        int64
}

type tokenResponse struct {
	Data struct {
		SecurityToken   string `json:"securityToken"`
		AccessKeySecret string `json:"accessKeySecret"`
		AccessKeyId     string `json:"accessKeyId"`
		Expiration      string `json:"expiration"`
	} `json:"data"`
	Code    int    `json:"code"`
	Message string `json:"message"`
}

func (fs *FileService) GetToken() (ossCredentials, error) {
	now := time.Now().UnixMilli()
	if v, ok := cache.Load(credentailKey); ok {
		c := v.(ossCredentials)
		// 预留2分钟缓冲时间
		if c.deadline-now >= 120000 {
			return c, nil
		}
		// 过期后删除
		cache.Delete(credentailKey)
	}

	// 获取STS token
	value, err, _ := group.Do(credentailKey, func() (interface{}, error) {
		url := fs.config.LbkOssApiHost + tokenPath
		resp, err := http.Get(url)
		if err != nil {
			return nil, err
		}
		defer resp.Body.Close()

		if resp.StatusCode != http.StatusOK {
			return nil, fmt.Errorf("api host response code is: %d", resp.StatusCode)
		}

		body, err := io.ReadAll(resp.Body)
		if err != nil {
			return nil, err
		}

		var response tokenResponse
		err = json.Unmarshal(body, &response)
		if err != nil {
			return nil, err
		}
		deadline, err := strconv.ParseInt(response.Data.Expiration, 10, 64)
		if err != nil {
			return nil, err
		}

		c := ossCredentials{
			accessKeyID:     response.Data.AccessKeyId,
			accessKeySecret: response.Data.AccessKeySecret,
			securityToken:   response.Data.SecurityToken,
			deadline:        deadline,
		}
		cache.Store(credentailKey, c)

		return c, nil
	})
	if err != nil {
		return ossCredentials{}, err
	}
	c := value.(ossCredentials)

	return c, nil
}
